import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import axios from 'axios';
import * as nodemailer from 'nodemailer';

@Injectable()
export class StockAlertTelegramService {
  private logger = new Logger('StockAlertTelegramService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি ঘণ্টায় স্টক চেক করুন
  @Cron('0 * * * *')
  async checkLowStockAndAlert() {
    this.logger.log('Checking for low stock products...');

    const alerts = await this.prisma.stockAlert.findMany({
      where: { isActive: true },
      include: { product: true }
    });

    for (const alert of alerts) {
      if (alert.product.stock <= alert.threshold) {
        try {
          // টেলিগ্রাম এবং ইমেইল দোনো পাঠান
          await this.sendTelegramAlert(alert);
          await this.sendEmailAlert(alert);

          // অ্যালার্ট মার্ক করুন
          await this.prisma.stockAlert.update({
            where: { id: alert.id },
            data: { alertSent: true }
          });
        } catch (err) {
          this.logger.error(`Stock alert failed: ${alert.id}:`, err);
        }
      }
    }
  }

  private async sendTelegramAlert(alert: any) {
    const message = `🚨 LOW STOCK ALERT\n\nProduct: ${alert.product.title}\nCurrent Stock: ${alert.product.stock}\nThreshold: ${alert.threshold}\n\nPlease reorder immediately!`;

    try {
      await axios.post(
        `https://api.telegram.org/bot${process.env.TELEGRAM_BOT_TOKEN}/sendMessage`,
        {
          chat_id: process.env.TELEGRAM_CHAT_ID,
          text: message
        }
      );

      this.logger.log(`Telegram alert sent for product ${alert.productId}`);
    } catch (err) {
      this.logger.error('Telegram alert failed:', err);
    }
  }

  private async sendEmailAlert(alert: any) {
    const emailHtml = `
<h2>⚠️ Low Stock Alert</h2>
<p>Product: <strong>${alert.product.title}</strong></p>
<p>Current Stock: <strong>${alert.product.stock}</strong></p>
<p>Alert Threshold: ${alert.threshold}</p>
<p>Please reorder this product immediately to avoid stockouts!</p>
    `;

    // অ্যাডমিনের ইমেইল খুঁজুন
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: alert.product.tenantId }
    });

    if (tenant) {
      await this.mailer.sendMail({
        from: process.env.STORE_EMAIL || 'noreply@store.com',
        to: tenant.email,
        subject: `⚠️ Low Stock: ${alert.product.title}`,
        html: emailHtml
      });
    }
  }
}