import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class StockAlertAutomationService {
  private logger = new Logger('StockAlertAutomationService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি ঘণ্টায় স্টক চেক করুন
  @Cron('0 * * * *')
  async checkLowStockProducts() {
    this.logger.log('Checking for low stock products...');

    const alerts = await this.prisma.stockAlert.findMany({
      where: { isActive: true, alertSent: false },
      include: { product: true }
    });

    for (const alert of alerts) {
      if (alert.product.stock <= alert.threshold) {
        try {
          await this.sendStockAlert(alert);
        } catch (err) {
          this.logger.error(`Stock alert failed for product ${alert.productId}:`, err);
        }
      }
    }
  }

  private async sendStockAlert(alert: any) {
    // স্টোর অ্যাডমিনকে নোটিফাই করুন
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: alert.product.tenantId }
    });

    if (!tenant) return;

    const emailHtml = `
<h2>Low Stock Alert!</h2>
<p>Product: ${alert.product.title}</p>
<p>Current Stock: ${alert.product.stock}</p>
<p>Alert Threshold: ${alert.threshold}</p>
<p>Please reorder this product immediately.</p>
    `;

    // অ্যাডমিনের ইমেইল পান
    const adminEmail = tenant.email;

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL || 'noreply@store.com',
      to: adminEmail,
      subject: `Low Stock Alert - ${alert.product.title}`,
      html: emailHtml
    });

    // সতর্কতা মার্ক করুন
    await this.prisma.stockAlert.update({
      where: { id: alert.id },
      data: { alertSent: true }
    });

    this.logger.log(`Stock alert sent for product ${alert.productId}`);
  }

  // স্টক সতর্কতা তৈরি করুন
  async createStockAlert(tenantId: string, productId: string, threshold: number) {
    const alert = await this.prisma.stockAlert.create({
      data: {
        tenantId,
        productId,
        threshold
      }
    });

    return { success: true, data: alert };
  }
}