import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class ThemeBackupAutomationService {
  private logger = new Logger('ThemeBackupAutomationService');

  constructor(private prisma: PrismaService) {}

  // থিম পাবলিশ করার আগে ব্যাকআপ করুন
  async backupBeforePublish(tenantId: string, themeConfig: any, themeName: string) {
    try {
      // সর্বোচ্চ সংস্করণ খুঁজুন
      const latestBackup = await this.prisma.themeBackup.findFirst({
        where: { tenantId },
        orderBy: { version: 'desc' }
      });

      const newVersion = (latestBackup?.version || 0) + 1;

      // ব্যাকআপ তৈরি করুন
      const backup = await this.prisma.themeBackup.create({
        data: {
          tenantId,
          version: newVersion,
          themeName,
          themeConfig,
          backupReason: 'pre_publish'
        }
      });

      this.logger.log(
        `Theme backup created: v${newVersion} for tenant ${tenantId}`
      );

      return {
        success: true,
        backupVersion: newVersion,
        canRollback: true
      };
    } catch (err) {
      this.logger.error('Theme backup failed:', err);
      throw err;
    }
  }

  // আগের থিম সংস্করণে রোলব্যাক করুন
  async rollbackTheme(tenantId: string, toVersion: number) {
    try {
      const backup = await this.prisma.themeBackup.findFirst({
        where: {
          tenantId,
          version: toVersion
        }
      });

      if (!backup) throw new Error('Backup version not found');

      // পুরানো কনফিগ রিস্টোর করুন
      // TODO: Apply theme configuration

      this.logger.log(`Theme rolled back to v${toVersion}`);

      return { success: true, restoredVersion: toVersion };
    } catch (err) {
      this.logger.error('Theme rollback failed:', err);
      throw err;
    }
  }

  // থিম সংস্করণ ইতিহাস পান
  async getThemeVersionHistory(tenantId: string) {
    const backups = await this.prisma.themeBackup.findMany({
      where: { tenantId },
      orderBy: { version: 'desc' }
    });

    return {
      versions: backups.map((b) => ({
        version: b.version,
        theme: b.themeName,
        createdAt: b.createdAt,
        reason: b.backupReason
      }))
    };
  }
}