import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class ProductTranslationService {
  private logger = new Logger('ProductTranslationService');
  private supportedLanguages = ['es', 'fr', 'de', 'ja', 'ar', 'pt', 'ru', 'zh'];

  constructor(private prisma: PrismaService) {}

  // Google Translate API দিয়ে অনুবাদ করুন
  async translateProductContent(
    productId: string,
    targetLanguages?: string[]
  ) {
    const product = await this.prisma.product.findUnique({
      where: { id: productId }
    });

    if (!product) throw new Error('Product not found');

    const languages = targetLanguages || this.supportedLanguages;

    for (const lang of languages) {
      try {
        // Google Translate API কল করুন
        const translatedTitle = await this.translateText(product.title, lang);
        const translatedDesc = await this.translateText(
          product.description || '',
          lang
        );

        // অনুবাদ সংরক্ষণ করুন
        await this.prisma.productTranslation.upsert({
          where: {
            productId_language: { productId, language: lang }
          },
          create: {
            productId,
            language: lang,
            title: translatedTitle,
            description: translatedDesc
          },
          update: {
            title: translatedTitle,
            description: translatedDesc,
            lastSyncedAt: new Date()
          }
        });

        this.logger.log(`Product translated to ${lang}: ${product.id}`);
      } catch (err) {
        this.logger.error(`Translation failed for ${lang}:`, err);
      }
    }

    return { success: true };
  }

  private async translateText(text: string, targetLang: string): Promise<string> {
    try {
      const response = await axios.post(
        'https://translation.googleapis.com/language/translate/v2',
        {
          q: text,
          target: targetLang,
          source: 'en'
        },
        {
          params: {
            key: process.env.GOOGLE_TRANSLATE_API_KEY
          }
        }
      );

      return response.data.data.translations[0].translatedText;
    } catch (err) {
      this.logger.error('Translation API call failed:', err);
      return text; // ফেইল হলে অরিজিনাল টেক্সট রিটার্ন করুন
    }
  }

  // অনুবাদিত পণ্য পান
  async getTranslatedProduct(productId: string, language: string) {
    const translation = await this.prisma.productTranslation.findUnique({
      where: { productId_language: { productId, language } }
    });

    return {
      translation: translation || {
        title: '',
        description: ''
      }
    };
  }

  // বাল্ক অনুবাদ (সব পণ্য)
  async translateAllProducts(tenantId: string) {
    const products = await this.prisma.product.findMany({
      where: { tenantId }
    });

    for (const product of products) {
      await this.translateProductContent(product.id);
    }

    return { success: true, translatedCount: products.length };
  }
}