import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class WebhookAutomationService {
  private logger = new Logger('WebhookAutomationService');

  constructor(private prisma: PrismaService) {}

  // ওয়েবহুক ইভেন্ট ট্রিগার করুন
  async triggerWebhook(tenantId: string, event: string, data: any) {
    try {
      const webhooks = await this.prisma.automationWorkflow.findMany({
        where: {
          tenantId,
          type: 'webhook',
          trigger: event,
          isActive: true
        }
      });

      for (const webhook of webhooks) {
        const webhookUrl = webhook.config?.url;

        if (!webhookUrl) continue;

        try {
          await axios.post(webhookUrl, data, {
            timeout: 5000,
            headers: {
              'Content-Type': 'application/json',
              'X-Webhook-Signature': this.generateSignature(data)
            }
          });

          this.logger.log(`Webhook triggered: ${event}`);
        } catch (err) {
          this.logger.warn(`Webhook delivery failed for ${webhookUrl}:`, err);
        }
      }
    } catch (err) {
      this.logger.error('Webhook trigger failed:', err);
    }
  }

  private generateSignature(data: any): string {
    // HMAC স্বাক্ষর তৈরি করুন (নিরাপত্তার জন্য)
    const crypto = require('crypto');
    const signature = crypto
      .createHmac('sha256', process.env.WEBHOOK_SECRET || '')
      .update(JSON.stringify(data))
      .digest('hex');

    return signature;
  }

  // Zapier সামঞ্জস্য পরীক্ষা করুন
  async testZapierIntegration(tenantId: string, zapierWebhookUrl: string) {
    const testData = {
      event: 'test',
      timestamp: new Date().toISOString(),
      data: { test: true }
    };

    try {
      const response = await axios.post(zapierWebhookUrl, testData);
      return { success: true, data: response.data };
    } catch (err) {
      return { success: false, error: (err as Error).message };
    }
  }
}