import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as AWS from 'aws-sdk';
import * as fs from 'fs';

@Injectable()
export class BackupService {
  private logger = new Logger('BackupService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // টেন্যান্ট ব্যাকআপ করুন
  async backupTenant(tenantId: string) {
    try {
      this.logger.log(`Starting backup for tenant: ${tenantId}`);

      // 1. Export data থেকে database
      const data = await this.exportTenantData(tenantId);

      // 2. JSON ফাইল তৈরি করুন
      const filename = `backup-${tenantId}-${Date.now()}.json`;
      const filepath = `/tmp/${filename}`;
      fs.writeFileSync(filepath, JSON.stringify(data, null, 2));

      // 3. Upload to S3
      const s3Key = `backups/${tenantId}/${filename}`;
      await this.s3
        .putObject({
          Bucket: process.env.AWS_S3_BUCKET,
          Key: s3Key,
          Body: fs.readFileSync(filepath),
          ContentType: 'application/json'
        })
        .promise();

      // 4. Clean up local file
      fs.unlinkSync(filepath);

      // 5. Save backup record
      await this.prisma.backup.create({
        data: {
          tenantId,
          s3Key,
          filename,
          dataSize: fs.statSync(filepath).size,
          status: 'completed'
        }
      });

      this.logger.log(`Backup completed for tenant: ${tenantId}`);

      return { success: true, filename };
    } catch (err) {
      this.logger.error(`Backup failed for tenant ${tenantId}:`, err);
      throw err;
    }
  }

  // টেন্যান্ট রিস্টোর করুন
  async restoreTenant(tenantId: string, backupId: string) {
    try {
      this.logger.log(`Starting restore for tenant: ${tenantId}`);

      // 1. Get backup from S3
      const backup = await this.prisma.backup.findUnique({
        where: { id: backupId }
      });

      if (!backup) throw new Error('Backup not found');

      const s3Object = await this.s3
        .getObject({
          Bucket: process.env.AWS_S3_BUCKET,
          Key: backup.s3Key
        })
        .promise();

      // 2. Parse data
      const data = JSON.parse(s3Object.Body.toString());

      // 3. Restore to database
      await this.restoreTenantData(tenantId, data);

      // 4. Update backup status
      await this.prisma.backup.update({
        where: { id: backupId },
        data: { status: 'restored', restoredAt: new Date() }
      });

      this.logger.log(`Restore completed for tenant: ${tenantId}`);

      return { success: true };
    } catch (err) {
      this.logger.error(`Restore failed for tenant ${tenantId}:`, err);
      throw err;
    }
  }

  // মাসিক বা সাপ্তাহিক অটো-ব্যাকআপ
  @Cron('0 2 * * *') // প্রতিদিন রাত ২ টায়
  async automaticBackup() {
    const tenants = await this.prisma.tenant.findMany({
      where: { status: 'ACTIVE' }
    });

    for (const tenant of tenants) {
      await this.backupTenant(tenant.id);
    }
  }

  private async exportTenantData(tenantId: string) {
    return {
      products: await this.prisma.product.findMany({ where: { tenantId } }),
      orders: await this.prisma.order.findMany({ where: { tenantId } }),
      customers: await this.prisma.user.findMany({ where: { tenantId } }),
      settings: await this.prisma.tenantSettings.findUnique({ where: { tenantId } }),
      timestamp: new Date().toISOString()
    };
  }

  private async restoreTenantData(tenantId: string, data: any) {
    // Restore data to database
    // Note: This is simplified; actual restore would need transaction handling
    for (const product of data.products) {
      await this.prisma.product.upsert({
        where: { id: product.id },
        update: product,
        create: product
      });
    }
  }
}