import { Injectable, BadRequestException, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import Stripe from 'stripe';

@Injectable()
export class BillingService {
  private stripe: Stripe;
  private logger = new Logger('BillingService');

  constructor(private prisma: PrismaService) {
    this.stripe = new Stripe(process.env.STRIPE_SECRET_KEY, {
      apiVersion: '2023-10-16'
    });
  }

  // Plan pricing (configurable)
  private plans = {
    FREE: { name: 'Free', price: 0, features: ['5 products', '1 user'] },
    STARTER: { name: 'Starter', price: 2999, features: ['100 products', '3 users', 'Email support'] },
    PROFESSIONAL: { name: 'Professional', price: 9999, features: ['Unlimited products', '10 users', 'Priority support'] },
    ENTERPRISE: { name: 'Enterprise', price: 29999, features: ['Everything', 'Dedicated support', 'API access'] }
  };

  async createCheckoutSession(tenantId: string, planType: string) {
    const tenant = await this.prisma.tenant.findUnique({ where: { id: tenantId } });

    if (!tenant) {
      throw new BadRequestException('Tenant not found');
    }

    const plan = this.plans[planType];
    if (!plan) {
      throw new BadRequestException('Invalid plan');
    }

    // Get or create Stripe customer
    let stripeCustomerId = null;
    const existingSub = await this.prisma.subscription.findUnique({
      where: { tenantId_plan: { tenantId, plan: planType } }
    });

    if (existingSub?.stripeCustomerId) {
      stripeCustomerId = existingSub.stripeCustomerId;
    } else {
      const customer = await this.stripe.customers.create({
        email: tenant.billingEmail || 'noreply@' + tenant.host,
        metadata: { tenantId }
      });
      stripeCustomerId = customer.id;
    }

    // Create checkout session
    const session = await this.stripe.checkout.sessions.create({
      customer: stripeCustomerId,
      payment_method_types: ['card'],
      line_items: [
        {
          price_data: {
            currency: tenant.currency.toLowerCase(),
            product_data: {
              name: plan.name,
              description: plan.features.join(', ')
            },
            unit_amount: plan.price
          },
          quantity: 1
        }
      ],
      mode: 'subscription',
      success_url: `${process.env.APP_URL}/dashboard?session_id={CHECKOUT_SESSION_ID}`,
      cancel_url: `${process.env.APP_URL}/billing`,
      metadata: {
        tenantId,
        planType
      }
    });

    return {
      success: true,
      data: {
        sessionId: session.id,
        checkoutUrl: session.url
      }
    };
  }

  async handleCheckoutComplete(sessionId: string) {
    const session = await this.stripe.checkout.sessions.retrieve(sessionId, {
      expand: ['subscription']
    });

    const { tenantId, planType } = session.metadata as any;
    const subscription = session.subscription as Stripe.Subscription;

    // Create or update subscription record
    const existingSub = await this.prisma.subscription.findUnique({
      where: { tenantId_plan: { tenantId, plan: planType } }
    });

    if (existingSub) {
      await this.prisma.subscription.update({
        where: { id: existingSub.id },
        data: {
          stripeSubId: subscription.id,
          status: 'ACTIVE',
          currentPeriodStart: new Date(subscription.current_period_start * 1000),
          currentPeriodEnd: new Date(subscription.current_period_end * 1000)
        }
      });
    } else {
      await this.prisma.subscription.create({
        data: {
          tenantId,
          plan: planType as any,
          stripeCustomerId: session.customer as string,
          stripeSubId: subscription.id,
          status: 'ACTIVE',
          currentPeriodStart: new Date(subscription.current_period_start * 1000),
          currentPeriodEnd: new Date(subscription.current_period_end * 1000)
        }
      });
    }

    // Update tenant plan
    await this.prisma.tenant.update({
      where: { id: tenantId },
      data: { plan: planType as any }
    });

    this.logger.log(`Subscription created for tenant ${tenantId} on plan ${planType}`);

    return { success: true };
  }

  async handleWebhook(event: Stripe.Event) {
    switch (event.type) {
      case 'checkout.session.completed':
        return await this.handleCheckoutComplete((event.data.object as Stripe.Checkout.Session).id);

      case 'customer.subscription.updated':
        const updatedSub = event.data.object as Stripe.Subscription;
        await this.prisma.subscription.updateMany({
          where: { stripeSubId: updatedSub.id },
          data: {
            currentPeriodStart: new Date(updatedSub.current_period_start * 1000),
            currentPeriodEnd: new Date(updatedSub.current_period_end * 1000)
          }
        });
        break;

      case 'customer.subscription.deleted':
        const deletedSub = event.data.object as Stripe.Subscription;
        await this.prisma.subscription.updateMany({
          where: { stripeSubId: deletedSub.id },
          data: { status: 'CANCELLED' }
        });
        break;

      case 'invoice.payment_succeeded':
        const invoice = event.data.object as Stripe.Invoice;
        this.logger.log(`Invoice paid: ${invoice.id}`);
        break;
    }

    return { received: true };
  }

  async getSubscriptionDetails(tenantId: string) {
    const subscription = await this.prisma.subscription.findFirst({
      where: { tenantId, status: 'ACTIVE' }
    });

    if (!subscription) {
      return { data: null };
    }

    return {
      data: {
        plan: subscription.plan,
        status: subscription.status,
        currentPeriodStart: subscription.currentPeriodStart,
        currentPeriodEnd: subscription.currentPeriodEnd,
        cancelledAt: subscription.cancelledAt
      }
    };
  }

  async cancelSubscription(tenantId: string) {
    const subscription = await this.prisma.subscription.findFirst({
      where: { tenantId, status: 'ACTIVE' }
    });

    if (!subscription || !subscription.stripeSubId) {
      throw new BadRequestException('No active subscription');
    }

    await this.stripe.subscriptions.del(subscription.stripeSubId);

    return { success: true, message: 'Subscription cancelled' };
  }
}