import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class PageBuilderService {
  private logger = new Logger('PageBuilderService');

  constructor(private prisma: PrismaService) {}

  // পেজ তৈরি করুন
  async createPage(tenantId: string, input: {
    name: string;
    type: string;
    content: any;
  }) {
    const page = await this.prisma.pageTemplate.create({
      data: {
        tenantId,
        name: input.name,
        type: input.type,
        content: input.content,
        isPublished: false
      }
    });

    this.logger.log(`Page created: ${page.id}`);

    return { success: true, data: page };
  }

  // ব্লক যোগ করুন
  async addBlock(pageId: string, input: {
    type: string;
    position: number;
    settings: any;
    content: string;
  }) {
    const block = await this.prisma.pageBlock.create({
      data: {
        pageId,
        type: input.type,
        position: input.position,
        settings: input.settings,
        content: input.content
      }
    });

    return { success: true, data: block };
  }

  // ব্লক আপডেট করুন
  async updateBlock(blockId: string, input: any) {
    const block = await this.prisma.pageBlock.update({
      where: { id: blockId },
      data: {
        settings: input.settings,
        content: input.content,
        position: input.position
      }
    });

    return { success: true, data: block };
  }

  // ব্লক ডিলিট করুন
  async deleteBlock(blockId: string) {
    await this.prisma.pageBlock.delete({ where: { id: blockId } });
    return { success: true };
  }

  // পেজ প্রকাশ করুন (backup সহ)
  async publishPage(pageId: string) {
    const page = await this.prisma.pageTemplate.findUnique({
      where: { id: pageId }
    });

    if (!page) throw new BadRequestException('Page not found');

    // পূর্ববর্তী সংস্করণ backup হিসেবে সংরক্ষণ
    const updated = await this.prisma.pageTemplate.update({
      where: { id: pageId },
      data: {
        isPublished: true,
        publishedAt: new Date(),
        backup: JSON.stringify(page.content),
        version: page.version + 1
      }
    });

    this.logger.log(`Page published: ${pageId}`);

    return { success: true, data: updated };
  }

  // পূর্ববর্তী সংস্করণ পুনরুদ্ধার করুন
  async restorePreviousVersion(pageId: string) {
    const page = await this.prisma.pageTemplate.findUnique({
      where: { id: pageId }
    });

    if (!page?.backup) throw new BadRequestException('No backup available');

    const restored = await this.prisma.pageTemplate.update({
      where: { id: pageId },
      data: {
        content: JSON.parse(page.backup),
        backup: JSON.stringify(page.content),
        version: page.version - 1
      }
    });

    return { success: true, data: restored };
  }
}