import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CartService {
  constructor(private prisma: PrismaService) {}

  // কার্টে যোগ করুন
  async addToCart(
    userId: string | null,
    sessionId: string | null,
    tenantId: string,
    productId: string,
    quantity: number
  ) {
    const cartSession = await this.prisma.cartSession.findFirst({
      where: {
        tenantId,
        ...(userId ? { userId } : { sessionId })
      }
    });

    const items = cartSession?.items || [];
    const existingItem = items.find((item: any) => item.productId === productId);

    let updatedItems;
    if (existingItem) {
      existingItem.quantity += quantity;
      updatedItems = items;
    } else {
      const product = await this.prisma.product.findUnique({
        where: { id: productId }
      });
      items.push({
        productId,
        quantity,
        price: product?.price || 0
      });
      updatedItems = items;
    }

    // মোট এমাউন্ট ক্যালকুলেট করুন
    const totalAmount = updatedItems.reduce(
      (sum: number, item: any) => sum + item.price * item.quantity,
      0
    );

    const cart = await this.prisma.cartSession.upsert({
      where: { id: cartSession?.id || 'new' },
      create: {
        tenantId,
        ...(userId ? { userId } : { sessionId }),
        items: updatedItems,
        totalAmount
      },
      update: {
        items: updatedItems,
        totalAmount,
        updatedAt: new Date()
      }
    });

    return { success: true, cart };
  }

  // কার্ট পান
  async getCart(userId: string | null, sessionId: string | null, tenantId: string) {
    const cart = await this.prisma.cartSession.findFirst({
      where: {
        tenantId,
        ...(userId ? { userId } : { sessionId })
      }
    });

    return { cart: cart || { items: [], totalAmount: 0 } };
  }

  // কার্ট থেকে সরান
  async removeFromCart(userId: string | null, sessionId: string | null, productId: string, tenantId: string) {
    const cart = await this.prisma.cartSession.findFirst({
      where: {
        tenantId,
        ...(userId ? { userId } : { sessionId })
      }
    });

    if (!cart) return { success: false };

    const updatedItems = cart.items.filter(
      (item: any) => item.productId !== productId
    );

    const totalAmount = updatedItems.reduce(
      (sum: number, item: any) => sum + item.price * item.quantity,
      0
    );

    await this.prisma.cartSession.update({
      where: { id: cart.id },
      data: { items: updatedItems, totalAmount }
    });

    return { success: true };
  }

  // কার্ট খালি করুন
  async clearCart(userId: string | null, sessionId: string | null, tenantId: string) {
    const cart = await this.prisma.cartSession.findFirst({
      where: {
        tenantId,
        ...(userId ? { userId } : { sessionId })
      }
    });

    if (!cart) return { success: false };

    await this.prisma.cartSession.update({
      where: { id: cart.id },
      data: { items: [], totalAmount: 0 }
    });

    return { success: true };
  }
}