import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CodeEditorService {
  constructor(private prisma: PrismaService) {}

  // কোড পান
  async getCode(tenantId: string, type: 'css' | 'js' | 'html') {
    const latest = await this.prisma.codeEditorHistory.findFirst({
      where: { tenantId, type },
      orderBy: { version: 'desc' }
    });

    return { code: latest?.code || '' };
  }

  // কোড সংরক্ষণ করুন
  async saveCode(
    tenantId: string,
    type: 'css' | 'js' | 'html',
    code: string,
    userId: string,
    reason?: string
  ) {
    // সর্বোচ্চ ভার্সন খুঁজুন
    const latest = await this.prisma.codeEditorHistory.findFirst({
      where: { tenantId, type },
      orderBy: { version: 'desc' }
    });

    const newVersion = (latest?.version || 0) + 1;

    // নতুন ভার্সন সংরক্ষণ করুন
    const history = await this.prisma.codeEditorHistory.create({
      data: {
        tenantId,
        type,
        code,
        version: newVersion,
        changedBy: userId,
        changeReason: reason
      }
    });

    return { success: true, version: newVersion };
  }

  // কোড হিস্টরি পান
  async getCodeHistory(tenantId: string, type: 'css' | 'js' | 'html') {
    return this.prisma.codeEditorHistory.findMany({
      where: { tenantId, type },
      orderBy: { version: 'desc' },
      take: 50
    });
  }

  // রোলব্যাক করুন
  async rollbackToVersion(
    tenantId: string,
    type: 'css' | 'js' | 'html',
    version: number,
    userId: string
  ) {
    const previousVersion = await this.prisma.codeEditorHistory.findFirst({
      where: { tenantId, type, version }
    });

    if (!previousVersion) throw new Error('Version not found');

    // নতুন এন্ট্রি তৈরি করুন
    const latest = await this.prisma.codeEditorHistory.findFirst({
      where: { tenantId, type },
      orderBy: { version: 'desc' }
    });

    const newVersion = (latest?.version || 0) + 1;

    await this.prisma.codeEditorHistory.create({
      data: {
        tenantId,
        type,
        code: previousVersion.code,
        version: newVersion,
        changedBy: userId,
        changeReason: `Rollback to v${version}`
      }
    });

    return { success: true, rolledBackTo: version };
  }
}