import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';
import * as cheerio from 'cheerio';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class ProductScraperService {
  private logger = new Logger('ProductScraperService');

  constructor(private prisma: PrismaService) {}

  // মাল্টিপল ওয়েবসাইট থেকে প্রোডাক্ট স্ক্র্যাপ করুন
  async scrapeProducts(
    tenantId: string,
    sourceConfig: {
      sourceUrl: string;          // যে সাইট থেকে স্ক্র্যাপ করব
      productSelector: string;     // CSS selector for products
      titleSelector: string;
      priceSelector: string;
      imageSelector: string;
      descriptionSelector: string;
      linkSelector: string;
      category: string;           // যে ক্যাটাগরিতে রাখব
      margin: number;            // markup percentage
    }
  ) {
    try {
      this.logger.log(`Scraping products from ${sourceConfig.sourceUrl}`);

      // ওয়েবসাইট থেকে HTML ফেচ করুন
      const response = await axios.get(sourceConfig.sourceUrl, {
        timeout: 10000,
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/91.0.4472.124'
        }
      });

      const $ = cheerio.load(response.data);
      const products: any[] = [];

      // সব প্রোডাক্ট এলিমেন্ট খুঁজুন
      $(sourceConfig.productSelector).each((index, element) => {
        try {
          const title = $(element).find(sourceConfig.titleSelector).text().trim();
          const priceText = $(element).find(sourceConfig.priceSelector).text().trim();
          const image = $(element).find(sourceConfig.imageSelector).attr('src');
          const description = $(element).find(sourceConfig.descriptionSelector).text().trim();
          const link = $(element).find(sourceConfig.linkSelector).attr('href');

          // প্রাইস এক্সট্র্যাক্ট করুন (সংখ্যা খুঁজুন)
          const price = parseFloat(priceText.replace(/[^\d.]/g, ''));

          if (title && price && image) {
            // Markup যোগ করুন (লাভ মার্জিন)
            const sellingPrice = price * (1 + sourceConfig.margin / 100);

            products.push({
              title,
              price: parseFloat(price.toFixed(2)),
              sellingPrice: parseFloat(sellingPrice.toFixed(2)),
              image,
              description: description || title,
              sourceLink: link || sourceConfig.sourceUrl,
              sourceName: new URL(sourceConfig.sourceUrl).hostname,
              category: sourceConfig.category
            });
          }
        } catch (err) {
          this.logger.warn(`Error parsing product ${index}:`, err);
        }
      });

      // স্ক্র্যাপ করা প্রোডাক্ট ডাটাবেসে সংরক্ষণ করুন
      const savedProducts = await Promise.all(
        products.map((product) =>
          this.prisma.product.upsert({
            where: {
              tenantId_externalSourceId: {
                tenantId,
                externalSourceId: `${product.sourceName}:${product.sourceLink}`
              }
            },
            create: {
              tenantId,
              title: product.title,
              description: product.description,
              price: product.sellingPrice,
              costPrice: product.price,
              images: [product.image],
              categoryId: sourceConfig.category,
              slug: this.generateSlug(product.title),
              externalSourceId: `${product.sourceName}:${product.sourceLink}`,
              externalSourceLink: product.sourceLink,
              isDropshipped: true
            },
            update: {
              price: product.sellingPrice,
              costPrice: product.price,
              images: [product.image],
              updatedAt: new Date()
            }
          })
        )
      );

      this.logger.log(`Scraped ${savedProducts.length} products from ${sourceConfig.sourceUrl}`);

      return {
        success: true,
        scrapedCount: savedProducts.length,
        products: savedProducts
      };
    } catch (err) {
      this.logger.error('Scraping failed:', err);
      throw err;
    }
  }

  // প্রতিদিন স্বয়ংক্রিয় স্ক্র্যাপিং
  @Cron('0 2 * * *') // প্রতিদিন রাত ২টায়
  async autoScrapeProducts() {
    this.logger.log('Starting auto scraping...');

    const scrappingConfigs = await this.prisma.scraperConfig.findMany({
      where: { isActive: true }
    });

    for (const config of scrappingConfigs) {
      try {
        await this.scrapeProducts(config.tenantId, config.config);
      } catch (err) {
        this.logger.error(`Auto scraping failed for tenant ${config.tenantId}:`, err);
      }
    }
  }

  private generateSlug(title: string): string {
    return title
      .toLowerCase()
      .replace(/[^\w\s-]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-');
  }
}