import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class EnterpriseService {
  private logger = new Logger('EnterpriseService');

  constructor(private prisma: PrismaService) {}

  // ===== AUDIT LOGS =====
  async createAuditLog(
    tenantId: string,
    userId: string,
    action: string,
    entity: string,
    entityId: string,
    changes?: any
  ) {
    const auditLog = await this.prisma.auditLog.create({
      data: {
        tenantId,
        userId,
        action,
        entity,
        entityId,
        changes,
        ipAddress: '0.0.0.0', // Request context থেকে পান
        userAgent: 'Mozilla/5.0' // Request header থেকে পান
      }
    });

    this.logger.log(`Audit: ${action} on ${entity} ${entityId}`);
    return auditLog;
  }

  async getAuditLogs(tenantId: string, days = 30) {
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    return this.prisma.auditLog.findMany({
      where: {
        tenantId,
        createdAt: { gte: startDate }
      },
      orderBy: { createdAt: 'desc' }
    });
  }

  // ===== CUSTOM SCRIPTS =====
  async injectCustomScript(
    tenantId: string,
    script: string,
    location: 'head' | 'footer' | 'body_start' | 'body_end',
    type: 'javascript' | 'css'
  ) {
    const customScript = await this.prisma.customScript.create({
      data: {
        tenantId,
        script,
        location,
        type,
        isActive: true
      }
    });

    return { success: true, scriptId: customScript.id };
  }

  async getCustomScripts(tenantId: string) {
    return this.prisma.customScript.findMany({
      where: { tenantId, isActive: true },
      orderBy: { priority: 'desc' }
    });
  }

  // ===== WHITE-LABEL =====
  async setupWhiteLabel(
    tenantId: string,
    config: {
      brandName: string;
      brandColor: string;
      brandLogo: string;
      customDomain?: string;
    }
  ) {
    const whiteLabelConfig = await this.prisma.whiteLabelConfig.upsert({
      where: { tenantId },
      create: {
        tenantId,
        ...config
      },
      update: config
    });

    return {
      success: true,
      message: 'White-label configured'
    };
  }

  // ===== API RATE LIMITS =====
  async setRateLimits(
    tenantId: string,
    planName: string,
    limits: {
      requestsPerHour: number;
      requestsPerDay: number;
      concurrentRequests: number;
    }
  ) {
    const rateLimit = await this.prisma.aPIRateLimit.upsert({
      where: {
        tenantId_planName: { tenantId, planName }
      },
      create: {
        tenantId,
        planName,
        ...limits
      },
      update: limits
    });

    return { success: true };
  }

  async getRateLimits(tenantId: string, planName: string) {
    return this.prisma.aPIRateLimit.findUnique({
      where: {
        tenantId_planName: { tenantId, planName }
      }
    });
  }

  // ===== RESOURCE MONITORING =====
  async recordResourceUsage(
    tenantId: string,
    usage: {
      cpuUsage: number;
      memoryUsage: number;
      storageUsage: number;
      apiCallsUsed: number;
      activeUsers: number;
    }
  ) {
    const record = await this.prisma.resourceUsage.create({
      data: {
        tenantId,
        ...usage,
        apiCallsLimit: 10000 // Plan থেকে পান
      }
    });

    return { success: true };
  }

  async getTenantResourceUsage(tenantId: string) {
    const latest = await this.prisma.resourceUsage.findFirst({
      where: { tenantId },
      orderBy: { recordedAt: 'desc' }
    });

    return latest;
  }

  async getResourceUsageTrend(tenantId: string, days = 7) {
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    return this.prisma.resourceUsage.findMany({
      where: {
        tenantId,
        recordedAt: { gte: startDate }
      },
      orderBy: { recordedAt: 'asc' }
    });
  }
}