import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CountryPaymentConfigService {
  private logger = new Logger('CountryPaymentConfigService');

  constructor(private prisma: PrismaService) {}

  // দেশের ডিফল্ট পেমেন্ট গেটওয়ে সেট করুন
  async setDefaultPaymentGateway(
    countryCode: string,
    gatewayName: string,
    config: any
  ) {
    const existing = await this.prisma.countryPaymentGateway.findFirst({
      where: {
        countryCode,
        isEnabled: true
      }
    });

    // পুরানো ডিফল্ট অক্ষম করুন
    if (existing) {
      await this.prisma.countryPaymentGateway.update({
        where: { id: existing.id },
        data: { priority: 1 } // দ্বিতীয় অগ্রাধিকার
      });
    }

    // নতুন ডিফল্ট সেট করুন
    const gateway = await this.prisma.countryPaymentGateway.upsert({
      where: {
        countryCode_gatewayName: { countryCode, gatewayName }
      },
      create: {
        countryCode,
        gatewayName,
        config,
        priority: 0, // সর্বোচ্চ অগ্রাধিকার
        isEnabled: true
      },
      update: {
        config,
        priority: 0,
        isEnabled: true
      }
    });

    this.logger.log(`Default payment gateway set: ${gatewayName} in ${countryCode}`);

    return { success: true, data: gateway };
  }

  // দেশের সব পেমেন্ট গেটওয়ে পান (অগ্রাধিকার অনুযায়ী)
  async getCountryPaymentMethods(countryCode: string) {
    const gateways = await this.prisma.countryPaymentGateway.findMany({
      where: {
        countryCode,
        isEnabled: true
      },
      orderBy: { priority: 'asc' }
    });

    return gateways.map((gateway) => ({
      id: gateway.id,
      name: gateway.gatewayName,
      isDefault: gateway.priority === 0,
      config: {
        // সংবেদনশীল তথ্য লুকান
        apiEndpoint: gateway.config?.apiEndpoint || '',
        publicKey: gateway.config?.publicKey || ''
      }
    }));
  }

  // পেমেন্ট প্রসেসিং রাউট করুন
  async routePaymentToGateway(
    countryCode: string,
    orderId: string,
    amount: number,
    currency: string
  ) {
    // ডিফল্ট গেটওয়ে পান
    const gateway = await this.prisma.countryPaymentGateway.findFirst({
      where: {
        countryCode,
        isEnabled: true,
        priority: 0
      }
    });

    if (!gateway) {
      throw new Error(`No default payment gateway configured for ${countryCode}`);
    }

    // গেটওয়ে-নির্দিষ্ট পেমেন্ট প্রসেস করুন
    const paymentResult = await this.processPaymentViaGateway(
      gateway.gatewayName,
      {
        orderId,
        amount,
        currency,
        config: gateway.config
      }
    );

    return paymentResult;
  }

  private async processPaymentViaGateway(
    gatewayName: string,
    paymentData: any
  ) {
    // Gateway-specific implementation
    this.logger.log(`Processing payment via ${gatewayName}`);

    // Return payment result
    return {
      success: true,
      transactionId: `TXN-${Date.now()}`,
      gateway: gatewayName
    };
  }

  // পেমেন্ট হেলথ মেট্রিক্স
  async getPaymentGatewayHealth(countryCode: string) {
    const gateways = await this.prisma.countryPaymentGateway.findMany({
      where: {
        countryCode,
        isEnabled: true
      }
    });

    const healthMetrics = await Promise.all(
      gateways.map(async (gateway) => {
        // সাফল্যের হার গণনা করুন
        // TODO: Calculate success rate from transaction logs

        return {
          gatewayName: gateway.gatewayName,
          successRate: 99.5,
          lastChecked: new Date(),
          status: 'healthy'
        };
      })
    );

    return healthMetrics;
  }
}