import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CountryWholesaleNetworkService {
  private logger = new Logger('CountryWholesaleNetworkService');

  constructor(private prisma: PrismaService) {}

  // দেশের জন্য হোলসেল নেটওয়ার্ক তৈরি করুন
  async createWholesaleNetwork(
    countryCode: string,
    networkConfig: {
      networkName: string;
      description?: string;
      productCategories: string[];
      minOrderValue: number;
      networkRules?: any;
    }
  ) {
    const network = await this.prisma.countryWholesaleNetwork.create({
      data: {
        countryCode,
        ...networkConfig
      }
    });

    this.logger.log(`Wholesale network created: ${networkConfig.networkName} in ${countryCode}`);

    return {
      success: true,
      networkId: network.id,
      countryCode
    };
  }

  // নেটওয়ার্কে হোলসেলার যুক্ত করুন
  async addWholesalerToNetwork(
    networkId: string,
    wholesalerId: string
  ) {
    const network = await this.prisma.countryWholesaleNetwork.findUnique({
      where: { id: networkId }
    });

    if (!network) throw new Error('Network not found');

    // হোলসেলার যুক্ত করুন
    await this.prisma.countryWholesaleNetwork.update({
      where: { id: networkId },
      data: {
        wholesalersCount: { increment: 1 }
      }
    });

    this.logger.log(`Wholesaler added to network: ${networkId}`);

    return { success: true };
  }

  // নেটওয়ার্কের পণ্য পান
  async getNetworkProducts(networkId: string) {
    const network = await this.prisma.countryWholesaleNetwork.findUnique({
      where: { id: networkId },
      include: {
        countryConfig: true
      }
    });

    if (!network) throw new Error('Network not found');

    // নেটওয়ার্কের পণ্য ফিল্টার করুন
    const products = await this.prisma.product.findMany({
      where: {
        countryCode: network.countryCode,
        categoryId: { in: network.productCategories }
      }
    });

    return { products, networkName: network.networkName };
  }

  // নেটওয়ার্ক পলিসি প্রয়োগ করুন
  async enforceNetworkPolicy(
    networkId: string,
    orderId: string
  ): Promise<boolean> {
    const network = await this.prisma.countryWholesaleNetwork.findUnique({
      where: { id: networkId }
    });

    if (!network) throw new Error('Network not found');

    const order = await this.prisma.order.findUnique({
      where: { id: orderId }
    });

    if (!order) throw new Error('Order not found');

    // ন্যূনতম অর্ডার মূল্য যাচাই করুন
    if (order.total < Number(network.minOrderValue)) {
      return false; // নীতি লঙ্ঘন
    }

    // কাস্টম নীতি প্রয়োগ করুন
    if (network.networkRules) {
      // TODO: Implement custom rules
    }

    return true;
  }

  // নেটওয়ার্ক ড্যাশবোর্ড পান
  async getNetworkDashboard(networkId: string) {
    const network = await this.prisma.countryWholesaleNetwork.findUnique({
      where: { id: networkId }
    });

    if (!network) throw new Error('Network not found');

    return {
      networkName: network.networkName,
      countryCode: network.countryCode,
      wholesalers: network.wholesalersCount,
      activeStores: network.activeStores,
      categories: network.productCategories,
      minOrderValue: network.minOrderValue
    };
  }
}