import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class CurrencyConversionService {
  private logger = new Logger('CurrencyConversionService');

  constructor(private prisma: PrismaService) {}

  // এক্সচেঞ্জ রেট আপডেট করুন
  async updateExchangeRates() {
    try {
      // ফ্রি এক্সচেঞ্জ রেট API ব্যবহার করুন
      const response = await axios.get(
        'https://api.exchangerate-api.com/v4/latest/USD'
      );

      const rates = response.data.rates;

      // প্রধান মুদ্রাগুলির জন্য রেট আপডেট করুন
      const currencies = ['BDT', 'INR', 'PKR', 'LKR', 'GBP', 'EUR', 'JPY'];

      for (const currency of currencies) {
        if (rates[currency]) {
          await this.prisma.currencyExchangeRate.upsert({
            where: {
              baseCurrency_targetCurrency: {
                baseCurrency: 'USD',
                targetCurrency: currency
              }
            },
            create: {
              baseCurrency: 'USD',
              targetCurrency: currency,
              rate: rates[currency],
              rateType: 'realtime'
            },
            update: {
              rate: rates[currency],
              lastUpdatedAt: new Date()
            }
          });
        }
      }

      this.logger.log('Exchange rates updated');
      return { success: true };
    } catch (err) {
      this.logger.error('Exchange rate update failed:', err);
      throw err;
    }
  }

  // মুদ্রা রূপান্তর করুন
  async convertCurrency(
    amount: number,
    fromCurrency: string,
    toCurrency: string
  ): Promise<number> {
    if (fromCurrency === toCurrency) {
      return amount; // কোনো রূপান্তরের প্রয়োজন নেই
    }

    // রেট পান
    const rate = await this.prisma.currencyExchangeRate.findUnique({
      where: {
        baseCurrency_targetCurrency: {
          baseCurrency: fromCurrency,
          targetCurrency: toCurrency
        }
      }
    });

    if (!rate) {
      // বিপরীত রূপান্তর চেষ্টা করুন
      const reverseRate = await this.prisma.currencyExchangeRate.findUnique({
        where: {
          baseCurrency_targetCurrency: {
            baseCurrency: toCurrency,
            targetCurrency: fromCurrency
          }
        }
      });

      if (!reverseRate) {
        throw new Error(
          `No exchange rate found for ${fromCurrency} to ${toCurrency}`
        );
      }

      return amount / Number(reverseRate.rate);
    }

    return amount * Number(rate.rate);
  }

  // স্টোরের জন্য মূল্য রূপান্তর করুন
  async convertProductPrice(
    productPrice: number,
    productCurrency: string,
    targetCountryCode: string
  ): Promise<number> {
    const targetCountry = await this.prisma.countryConfig.findUnique({
      where: { countryCode: targetCountryCode }
    });

    if (!targetCountry) throw new Error('Country not found');

    return this.convertCurrency(
      productPrice,
      productCurrency,
      targetCountry.currency
    );
  }

  // ফিক্সড রেট ব্যবহার করুন (ম্যানুয়াল কনফিগারেশন)
  async setFixedRate(
    baseCurrency: string,
    targetCurrency: string,
    rate: number
  ) {
    const exchangeRate = await this.prisma.currencyExchangeRate.upsert({
      where: {
        baseCurrency_targetCurrency: { baseCurrency, targetCurrency }
      },
      create: {
        baseCurrency,
        targetCurrency,
        rate,
        rateType: 'fixed'
      },
      update: {
        rate,
        rateType: 'fixed',
        lastUpdatedAt: new Date()
      }
    });

    return { success: true, data: exchangeRate };
  }
}