import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CustomScriptService {
  private logger = new Logger('CustomScriptService');

  constructor(private prisma: PrismaService) {}

  // কাস্টম স্ক্রিপ্ট যোগ করুন
  async addCustomScript(
    tenantId: string,
    scriptData: {
      scriptType: 'javascript' | 'css' | 'pixel' | 'gtm';
      scriptName: string;
      scriptCode: string;
      placement: 'head' | 'body' | 'footer';
      description?: string;
    }
  ) {
    const script = await this.prisma.customScript.create({
      data: {
        tenantId,
        ...scriptData,
        isActive: true
      }
    });

    this.logger.log(`Custom script added: ${scriptData.scriptName} (${scriptData.scriptType})`);

    return {
      success: true,
      scriptId: script.id
    };
  }

  // সব কাস্টম স্ক্রিপ্ট পান
  async getCustomScripts(tenantId: string) {
    return this.prisma.customScript.findMany({
      where: { tenantId, isActive: true },
      select: {
        id: true,
        scriptName: true,
        scriptType: true,
        placement: true,
        isActive: true
      }
    });
  }

  // স্ক্রিপ্ট কোড পান (ফ্রন্টেন্ডের জন্য)
  async getScriptCode(tenantId: string): Promise<{
    headScripts: string[];
    bodyScripts: string[];
    footerScripts: string[];
    css: string[];
  }> {
    const scripts = await this.prisma.customScript.findMany({
      where: { tenantId, isActive: true }
    });

    const result = {
      headScripts: [] as string[],
      bodyScripts: [] as string[],
      footerScripts: [] as string[],
      css: [] as string[]
    };

    scripts.forEach((script) => {
      if (script.scriptType === 'css') {
        result.css.push(script.scriptCode);
      } else {
        const wrappedCode = script.scriptType === 'gtm'
          ? this.wrapGTMScript(script.scriptCode)
          : script.scriptCode;

        if (script.placement === 'head') {
          result.headScripts.push(wrappedCode);
        } else if (script.placement === 'body') {
          result.bodyScripts.push(wrappedCode);
        } else if (script.placement === 'footer') {
          result.footerScripts.push(wrappedCode);
        }
      }
    });

    return result;
  }

  // স্ক্রিপ্ট আপডেট করুন
  async updateScript(
    scriptId: string,
    updates: any
  ) {
    const script = await this.prisma.customScript.update({
      where: { id: scriptId },
      data: {
        ...updates,
        version: { increment: 1 }
      }
    });

    return { success: true };
  }

  // স্ক্রিপ্ট অক্ষম করুন
  async disableScript(scriptId: string) {
    await this.prisma.customScript.update({
      where: { id: scriptId },
      data: { isActive: false }
    });

    return { success: true };
  }

  private wrapGTMScript(gtmId: string): string {
    return `
<!-- Google Tag Manager -->
<script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
})(window,document,'script','dataLayer','${gtmId}');</script>
<!-- End Google Tag Manager -->
    `;
  }
}