import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as dns from 'dns';
import { promisify } from 'util';

@Injectable()
export class DNSSetupService {
  private logger = new Logger('DNSSetupService');
  private resolveTxt = promisify(dns.resolveTxt);
  private resolveCname = promisify(dns.resolveCname);

  constructor(private prisma: PrismaService) {}

  // ডোমেইন যুক্ত করুন
  async addDomain(tenantId: string, domain: string) {
    const dnsConfig = await this.prisma.dNSConfiguration.create({
      data: {
        tenantId,
        domain,
        isVerified: false
      }
    });

    // DNS রেকর্ড পরামর্শ তৈরি করুন
    const instructions = this.generateDNSInstructions(domain, tenantId);

    this.logger.log(`Domain added: ${domain} for tenant ${tenantId}`);

    return {
      success: true,
      dnsConfig,
      instructions
    };
  }

  // DNS রেকর্ড ভ্যালিডেট করুন
  async validateDomain(tenantId: string): Promise<boolean> {
    const dnsConfig = await this.prisma.dNSConfiguration.findUnique({
      where: { tenantId }
    });

    if (!dnsConfig) throw new Error('DNS config not found');

    try {
      // TXT রেকর্ড যাচাই করুন
      const txtRecords = await this.resolveTxt(dnsConfig.domain);

      const verificationToken = `platform-verify=${tenantId}`;
      const isValid = txtRecords.some((record) =>
        record.join('').includes(verificationToken)
      );

      if (isValid) {
        // ডোমেইন যাচাই করুন
        await this.prisma.dNSConfiguration.update({
          where: { tenantId },
          data: {
            isVerified: true,
            verifiedAt: new Date()
          }
        });

        this.logger.log(`Domain verified: ${dnsConfig.domain}`);
        return true;
      }

      return false;
    } catch (err) {
      this.logger.error('Domain verification failed:', err);
      return false;
    }
  }

  // DNS নির্দেশনা তৈরি করুন
  private generateDNSInstructions(domain: string, tenantId: string): any {
    return {
      verification: {
        type: 'TXT',
        name: `_${tenantId}`,
        value: `platform-verify=${tenantId}`,
        instructions: `Your DNS provider এ যান এবং একটি নতুন TXT রেকর্ড যোগ করুন:`
      },
      cname: {
        type: 'CNAME',
        name: 'www',
        value: `cdn.platform.com`,
        instructions: `www সাবডোমেইনের জন্য CNAME রেকর্ড যোগ করুন:`
      },
      aRecord: {
        type: 'A',
        name: '@',
        value: '93.184.216.34', // Example IP
        instructions: `রুট ডোমেইনের জন্য A রেকর্ড যোগ করুন (বা CNAME ব্যবহার করুন):`
      }
    };
  }

  // ডোমেইনের স্ট্যাটাস পান
  async getDomainStatus(tenantId: string) {
    const dnsConfig = await this.prisma.dNSConfiguration.findUnique({
      where: { tenantId }
    });

    if (!dnsConfig) throw new Error('Domain not configured');

    return {
      domain: dnsConfig.domain,
      isVerified: dnsConfig.isVerified,
      verifiedAt: dnsConfig.verifiedAt,
      status: dnsConfig.isVerified ? 'verified' : 'pending_verification'
    };
  }

  // ডোমেইন সংযোগ পরীক্ষা করুন
  async testDomainConnection(domain: string): Promise<boolean> {
    try {
      const addresses = await promisify(dns.resolve4)(domain);
      return addresses.length > 0;
    } catch (err) {
      return false;
    }
  }
}