import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as AWS from 'aws-sdk';

@Injectable()
export class InfrastructureService {
  private logger = new Logger('InfrastructureService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // ===== CUSTOM SMTP =====
  async configureSMTP(
    tenantId: string,
    smtpConfig: {
      provider: string;
      host: string;
      port: number;
      username: string;
      password: string;
      fromEmail: string;
      fromName: string;
    }
  ) {
    const smtp = await this.prisma.sMTPConfiguration.upsert({
      where: { tenantId },
      create: { tenantId, ...smtpConfig },
      update: smtpConfig
    });

    return { success: true, data: smtp };
  }

  // ===== API TOKEN MANAGEMENT =====
  async createAPIToken(
    tenantId: string,
    name: string,
    permissions: string[],
    expiresInDays?: number
  ) {
    const token = this.generateSecureToken();
    const hashedToken = this.hashToken(token);

    const expiresAt = expiresInDays
      ? new Date(Date.now() + expiresInDays * 24 * 60 * 60 * 1000)
      : null;

    const apiToken = await this.prisma.aPIToken.create({
      data: {
        tenantId,
        token: hashedToken,
        name,
        permissions,
        expiresAt
      }
    });

    return {
      success: true,
      token: token, // শুধুমাত্র একবার দেখান
      tokenId: apiToken.id
    };
  }

  async revokeAPIToken(tokenId: string) {
    await this.prisma.aPIToken.update({
      where: { id: tokenId },
      data: { isActive: false }
    });

    return { success: true };
  }

  // ===== BACKUP & RESTORE =====
  async createBackup(tenantId: string, backupType: 'full' | 'incremental' = 'full') {
    try {
      // ডাটাবেস ডাম্প তৈরি করুন
      // TODO: Implement database dump

      const backupData = JSON.stringify({
        tenantId,
        timestamp: new Date(),
        type: backupType
      });

      // S3-এ আপলোড করুন
      const key = `backups/${tenantId}/${Date.now()}.json`;

      await this.s3
        .putObject({
          Bucket: process.env.AWS_S3_BUCKET || '',
          Key: key,
          Body: backupData
        })
        .promise();

      const backup = await this.prisma.backupRecord.create({
        data: {
          tenantId,
          backupType,
          backupUrl: `s3://${process.env.AWS_S3_BUCKET}/${key}`,
          size: BigInt(backupData.length),
          status: 'completed',
          expiresAt: new Date(Date.now() + 90 * 24 * 60 * 60 * 1000) // ৯০ দিন
        }
      });

      return { success: true, data: backup };
    } catch (err) {
      this.logger.error('Backup creation failed:', err);
      throw err;
    }
  }

  async restoreBackup(backupId: string) {
    const backup = await this.prisma.backupRecord.findUnique({
      where: { id: backupId }
    });

    if (!backup) throw new Error('Backup not found');

    // TODO: Implement restore logic

    return { success: true, message: 'Restore initiated' };
  }

  // ===== LOG VIEWER =====
  async getLogs(
    tenantId: string,
    filters?: {
      logType?: string;
      limit?: number;
      offset?: number;
    }
  ) {
    const logs = await this.prisma.systemLog.findMany({
      where: {
        tenantId,
        ...(filters?.logType && { logType: filters.logType })
      },
      orderBy: { createdAt: 'desc' },
      take: filters?.limit || 100,
      skip: filters?.offset || 0
    });

    return logs;
  }

  // ===== DATABASE USAGE MONITOR =====
  async getDatabaseUsage(tenantId: string) {
    const usage = await this.prisma.databaseUsage.findFirst({
      where: { tenantId },
      orderBy: { recordedAt: 'desc' }
    });

    if (!usage) return null;

    const storagePercent = (Number(usage.storageUsed) / Number(usage.storageLimit)) * 100;
    const queryPercent = (usage.queryCount / usage.queryLimit) * 100;

    return {
      storage: {
        used: Number(usage.storageUsed),
        limit: Number(usage.storageLimit),
        percentage: storagePercent,
        status: storagePercent > 90 ? 'warning' : 'normal'
      },
      queries: {
        used: usage.queryCount,
        limit: usage.queryLimit,
        percentage: queryPercent,
        status: queryPercent > 90 ? 'warning' : 'normal'
      }
    };
  }

  // ===== MAINTENANCE MODE =====
  async scheduleMaintenance(
    tenantId: string,
    startTime: Date,
    endTime: Date,
    reason: string,
    message?: string
  ) {
    const maintenance = await this.prisma.maintenanceSchedule.upsert({
      where: { tenantId },
      create: {
        tenantId,
        startTime,
        endTime,
        reason,
        message,
        isScheduled: true,
        isActive: false
      },
      update: {
        startTime,
        endTime,
        reason,
        message,
        isScheduled: true
      }
    });

    return { success: true, data: maintenance };
  }

  async getMaintenanceStatus(tenantId: string) {
    const maintenance = await this.prisma.maintenanceSchedule.findUnique({
      where: { tenantId }
    });

    if (!maintenance) return { isMaintenanceActive: false };

    const now = new Date();
    const isActive = now >= maintenance.startTime && now <= maintenance.endTime;

    const timeRemaining = maintenance.endTime.getTime() - now.getTime();
    const minutesRemaining = Math.floor(timeRemaining / (1000 * 60));

    return {
      isMaintenanceActive: isActive || maintenance.isActive,
      reason: maintenance.reason,
      message: maintenance.message,
      startTime: maintenance.startTime,
      endTime: maintenance.endTime,
      minutesRemaining: minutesRemaining > 0 ? minutesRemaining : 0
    };
  }

  // ===== UTILITY METHODS =====
  private generateSecureToken(): string {
    const crypto = require('crypto');
    return crypto.randomBytes(32).toString('hex');
  }

  private hashToken(token: string): string {
    const crypto = require('crypto');
    return crypto.createHash('sha256').update(token).digest('hex');
  }
}