import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class PlatformIntegrationService {
  private logger = new Logger('PlatformIntegrationService');

  constructor(private prisma: PrismaService) {}

  // Shopify ইন্টিগ্রেশন সংযোগ
  async connectShopify(tenantId: string, shopifyStoreUrl: string, apiKey: string) {
    try {
      // Shopify API টেস্ট করুন
      const response = await axios.get(`https://${shopifyStoreUrl}/admin/api/2024-01/shop.json`, {
        headers: { 'X-Shopify-Access-Token': apiKey }
      });

      if (!response.data.shop) {
        throw new BadRequestException('Invalid Shopify credentials');
      }

      const integration = await this.prisma.platformIntegration.upsert({
        where: { tenantId_platform: { tenantId, platform: 'shopify' } },
        create: {
          tenantId,
          platform: 'shopify',
          apiKey,
          isActive: true
        },
        update: {
          apiKey,
          isActive: true
        }
      });

      this.logger.log(`Shopify connected: ${tenantId}`);

      // পণ্য সিঙ্ক শুরু করুন
      await this.syncShopifyProducts(tenantId, shopifyStoreUrl, apiKey);

      return { success: true, data: integration };
    } catch (err) {
      this.logger.error('Shopify connection failed:', err);
      throw err;
    }
  }

  // Shopify থেকে পণ্য সিঙ্ক করুন
  private async syncShopifyProducts(tenantId: string, storeUrl: string, apiKey: string) {
    try {
      const response = await axios.get(
        `https://${storeUrl}/admin/api/2024-01/products.json?limit=250`,
        { headers: { 'X-Shopify-Access-Token': apiKey } }
      );

      for (const product of response.data.products) {
        await this.prisma.product.upsert({
          where: { tenantId_sku: { tenantId, sku: product.handle } },
          create: {
            tenantId,
            sku: product.handle,
            title: product.title,
            description: product.body_html,
            price: parseFloat(product.variants[0]?.price || 0),
            stock: product.variants[0]?.inventory_quantity || 0,
            images: product.images?.map((img: any) => img.src) || []
          },
          update: {
            title: product.title,
            description: product.body_html,
            price: parseFloat(product.variants[0]?.price || 0),
            stock: product.variants[0]?.inventory_quantity || 0
          }
        });
      }

      this.logger.log(`Synced ${response.data.products.length} products from Shopify`);
    } catch (err) {
      this.logger.error('Shopify sync failed:', err);
    }
  }

  // WooCommerce ইন্টিগ্রেশন সংযোগ
  async connectWooCommerce(
    tenantId: string,
    storeUrl: string,
    consumerKey: string,
    consumerSecret: string
  ) {
    try {
      // WooCommerce API টেস্ট করুন
      const auth = Buffer.from(`${consumerKey}:${consumerSecret}`).toString('base64');
      const response = await axios.get(`${storeUrl}/wp-json/wc/v3/products?per_page=1`, {
        headers: { Authorization: `Basic ${auth}` }
      });

      const integration = await this.prisma.platformIntegration.upsert({
        where: { tenantId_platform: { tenantId, platform: 'woocommerce' } },
        create: {
          tenantId,
          platform: 'woocommerce',
          apiKey: consumerKey,
          apiSecret: consumerSecret,
          isActive: true
        },
        update: {
          apiKey: consumerKey,
          apiSecret: consumerSecret,
          isActive: true
        }
      });

      // পণ্য সিঙ্ক শুরু করুন
      await this.syncWooCommerceProducts(tenantId, storeUrl, auth);

      return { success: true, data: integration };
    } catch (err) {
      this.logger.error('WooCommerce connection failed:', err);
      throw err;
    }
  }

  // AliExpress ড্রপশিপিং সংযোগ
  async connectAliExpress(tenantId: string, apiKey: string) {
    const integration = await this.prisma.platformIntegration.upsert({
      where: { tenantId_platform: { tenantId, platform: 'aliexpress' } },
      create: {
        tenantId,
        platform: 'aliexpress',
        apiKey,
        isActive: true
      },
      update: {
        apiKey,
        isActive: true
      }
    });

    return { success: true, data: integration };
  }

  // সব ইন্টিগ্রেশন পান
  async getIntegrations(tenantId: string) {
    const integrations = await this.prisma.platformIntegration.findMany({
      where: { tenantId },
      select: {
        id: true,
        platform: true,
        isActive: true,
        lastSyncedAt: true,
        syncStatus: true
      }
    });

    return { success: true, data: integrations };
  }

  // মাল্টি-প্ল্যাটফর্ম পণ্য আপডেট
  async updateProductOnAllPlatforms(tenantId: string, productId: string, changes: any) {
    const integrations = await this.prisma.platformIntegration.findMany({
      where: { tenantId, isActive: true }
    });

    for (const integration of integrations) {
      try {
        if (integration.platform === 'shopify') {
          // Shopify-এ আপডেট করুন
        } else if (integration.platform === 'woocommerce') {
          // WooCommerce-এ আপডেট করুন
        }
      } catch (err) {
        this.logger.warn(`Failed to update on ${integration.platform}`, err);
      }
    }

    return { success: true };
  }

  private async syncWooCommerceProducts(tenantId: string, storeUrl: string, auth: string) {
    try {
      let page = 1;
      let hasMore = true;

      while (hasMore) {
        const response = await axios.get(
          `${storeUrl}/wp-json/wc/v3/products?page=${page}&per_page=100`,
          { headers: { Authorization: `Basic ${auth}` } }
        );

        for (const product of response.data) {
          await this.prisma.product.upsert({
            where: { tenantId_sku: { tenantId, sku: product.sku } },
            create: {
              tenantId,
              sku: product.sku || product.id.toString(),
              title: product.name,
              description: product.description,
              price: parseFloat(product.price),
              stock: product.stock_quantity || 0,
              images: product.images?.map((img: any) => img.src) || []
            },
            update: {
              title: product.name,
              price: parseFloat(product.price),
              stock: product.stock_quantity || 0
            }
          });
        }

        hasMore = response.data.length === 100;
        page++;
      }
    } catch (err) {
      this.logger.error('WooCommerce sync failed:', err);
    }
  }
}