import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as AWS from 'aws-sdk';

@Injectable()
export class KYCService {
  private logger = new Logger('KYCService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // KYC যাচাইকরণ শুরু করুন
  async submitKYC(
    tenantId: string,
    input: {
      businessName: string;
      businessType: string;
      businessRegNo?: string;
      documentType: string;
      documentNumber: string;
      documentFile: Buffer;
      address: string;
      city: string;
      state?: string;
      zipCode: string;
      country: string;
    }
  ) {
    try {
      // ডকুমেন্ট S3-এ আপলোড করুন
      const docUrl = await this.uploadDocument(tenantId, input.documentFile);

      const kyc = await this.prisma.kycVerification.create({
        data: {
          tenantId,
          businessName: input.businessName,
          businessType: input.businessType,
          businessRegNo: input.businessRegNo,
          documentType: input.documentType,
          documentNumber: input.documentNumber,
          documentUrl: docUrl,
          address: input.address,
          city: input.city,
          state: input.state,
          zipCode: input.zipCode,
          country: input.country,
          status: 'pending'
        }
      });

      this.logger.log(`KYC submitted: ${tenantId}`);

      return { success: true, data: kyc };
    } catch (err) {
      this.logger.error('KYC submission failed:', err);
      throw err;
    }
  }

  // KYC অনুমোদন করুন
  async approveKYC(kycId: string, adminId: string) {
    const kyc = await this.prisma.kycVerification.update({
      where: { id: kycId },
      data: {
        status: 'approved',
        verifiedAt: new Date(),
        verifiedBy: adminId
      }
    });

    return { success: true, data: kyc };
  }

  private async uploadDocument(tenantId: string, fileBuffer: Buffer): Promise<string> {
    const key = `kyc/${tenantId}/${Date.now()}.pdf`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: fileBuffer,
        ContentType: 'application/pdf'
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }
}