import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class AutoLanguageService {
  private languageMap: any = {
    BD: ['bn', 'en'],
    IN: ['hi', 'en', 'ta', 'te'],
    PK: ['ur', 'en'],
    LK: ['si', 'ta', 'en'],
    US: ['en'],
    GB: ['en'],
    FR: ['fr', 'en'],
    DE: ['de', 'en']
  };

  constructor(private prisma: PrismaService) {}

  // ব্যবহারকারীর পছন্দের ভাষা অনুমান করুন
  async detectUserLanguage(
    userId: string,
    ipAddress?: string,
    browserLanguage?: string
  ): Promise<string> {
    // ইতিমধ্যে সংরক্ষিত পছন্দ চেক করুন
    const preference = await this.prisma.languagePreference.findFirst({
      where: { userId, autoDetect: true }
    });

    if (preference && preference.preferredLanguage) {
      return preference.preferredLanguage;
    }

    // ব্রাউজার ভাষা চেক করুন
    if (browserLanguage) {
      const detected = this.extractLanguageCode(browserLanguage);
      if (detected) {
        await this.saveLanguagePreference(userId, detected);
        return detected;
      }
    }

    // IP-ভিত্তিক লোকেশন থেকে দেশ সনাক্ত করুন
    if (ipAddress) {
      const country = await this.getCountryFromIP(ipAddress);
      const languages = this.languageMap[country] || ['en'];

      const preferredLanguage = languages[0];
      await this.saveLanguagePreference(userId, preferredLanguage);

      return preferredLanguage;
    }

    // ডিফল্ট ইংরেজি
    return 'en';
  }

  // ভাষা পছন্দ সংরক্ষণ করুন
  private async saveLanguagePreference(
    userId: string,
    language: string
  ) {
    await this.prisma.languagePreference.upsert({
      where: { userId },
      create: {
        userId,
        preferredLanguage: language,
        autoDetect: true
      },
      update: {
        preferredLanguage: language
      }
    });
  }

  // ব্রাউজার ভাষা কোড এক্সট্র্যাক্ট করুন
  private extractLanguageCode(browserLanguage: string): string | null {
    // "bn-BD" থেকে "bn" বের করুন
    const code = browserLanguage.split('-')[0].toLowerCase();

    const supportedLanguages = ['en', 'bn', 'hi', 'ur', 'ta', 'fr', 'de', 'es'];

    return supportedLanguages.includes(code) ? code : null;
  }

  // IP থেকে দেশ পান (মকড আপ)
  private async getCountryFromIP(ipAddress: string): Promise<string> {
    // IP Geolocation API ব্যবহার করুন
    // বর্তমানে মকড করা হয়েছে
    return 'US';
  }

  // দেশের জন্য সমর্থিত ভাষা পান
  async getSupportedLanguages(countryCode: string): Promise<string[]> {
    return this.languageMap[countryCode] || ['en'];
  }
}