import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class TimezoneService {
  private timezoneMap: any = {
    BD: 'Asia/Dhaka',
    IN: 'Asia/Kolkata',
    PK: 'Asia/Karachi',
    US: 'America/New_York',
    GB: 'Europe/London',
    AU: 'Australia/Sydney',
    JP: 'Asia/Tokyo',
    SG: 'Asia/Singapore'
  };

  constructor(private prisma: PrismaService) {}

  // দেশের টাইমজোন পান
  async getCountryTimezone(countryCode: string): Promise<string> {
    return this.timezoneMap[countryCode] || 'UTC';
  }

  // টেন্যান্টের জন্য টাইমজোন সেট করুন
  async setTenantTimezone(tenantId: string, timezone: string) {
    const config = await this.prisma.timezoneConfig.upsert({
      where: { tenantId },
      create: {
        tenantId,
        timezone,
        offset: this.calculateUTCOffset(timezone)
      },
      update: {
        timezone,
        offset: this.calculateUTCOffset(timezone)
      }
    });

    return { success: true, data: config };
  }

  // সময় স্থানীয়করণ করুন
  localizeDateTime(date: Date, timezone: string): string {
    const formatter = new Intl.DateTimeFormat('en-US', {
      timeZone: timezone,
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit'
    });

    return formatter.format(date);
  }

  // রিপোর্টের জন্য টাইমজোন-সচেতন ডেটা পান
  async getLocalizedReportData(
    tenantId: string,
    startDate: Date,
    endDate: Date
  ) {
    const timezoneConfig = await this.prisma.timezoneConfig.findUnique({
      where: { tenantId }
    });

    const timezone = timezoneConfig?.timezone || 'UTC';

    // ডেটা সংগ্রহ করুন এবং টাইমজোন অনুযায়ী স্থানীয়করণ করুন
    // TODO: Fetch and localize data

    return {
      timezone,
      startDate: this.localizeDateTime(startDate, timezone),
      endDate: this.localizeDateTime(endDate, timezone)
    };
  }

  private calculateUTCOffset(timezone: string): number {
    const formatter = new Intl.DateTimeFormat('en-US', {
      timeZone: timezone,
      hour: '2-digit',
      hour12: false
    });

    const utcDate = new Date('2025-01-01T00:00:00Z');
    const localTime = new Date(
      formatter.format(utcDate).split(':')[0] + ':00:00'
    ).getTime();

    return (localTime - utcDate.getTime()) / (1000 * 60);
  }
}