import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class LoyaltyService {
  private logger = new Logger('LoyaltyService');

  constructor(private prisma: PrismaService) {}

  // অর্ডারে পয়েন্ট যোগ করুন
  async addPointsForOrder(userId: string, tenantId: string, orderAmount: number) {
    const pointsToAdd = Math.floor(orderAmount); // $1 = 1 point

    const loyaltyPoints = await this.prisma.loyaltyPoints.upsert({
      where: { userId_tenantId: { userId, tenantId } },
      create: {
        userId,
        tenantId,
        totalPoints: pointsToAdd
      },
      update: {
        totalPoints: { increment: pointsToAdd }
      }
    });

    // ট্রানজ্যাকশন রেকর্ড করুন
    await this.prisma.pointsTransaction.create({
      data: {
        userId,
        points: pointsToAdd,
        type: 'earned',
        description: `Earned from order`
      }
    });

    this.logger.log(`${pointsToAdd} points added to user ${userId}`);

    return { success: true, pointsAdded: pointsToAdd };
  }

  // পয়েন্ট রিডিম করুন
  async redeemPoints(userId: string, tenantId: string, points: number) {
    const loyaltyPoints = await this.prisma.loyaltyPoints.findUnique({
      where: { userId_tenantId: { userId, tenantId } }
    });

    if (!loyaltyPoints || loyaltyPoints.totalPoints - loyaltyPoints.usedPoints < points) {
      throw new Error('Insufficient points');
    }

    // পয়েন্ট ব্যবহার করুন
    await this.prisma.loyaltyPoints.update({
      where: { userId_tenantId: { userId, tenantId } },
      data: { usedPoints: { increment: points } }
    });

    // ট্রানজ্যাকশন রেকর্ড করুন
    await this.prisma.pointsTransaction.create({
      data: {
        userId,
        points: -points,
        type: 'redeemed',
        description: `Redeemed ${points} points`
      }
    });

    // রিওয়ার্ড ক্যালকুলেট করুন
    const reward = points / 100; // 100 points = $1 discount

    return { success: true, reward };
  }

  // পয়েন্ট ব্যালেন্স পান
  async getPointsBalance(userId: string, tenantId: string) {
    const loyaltyPoints = await this.prisma.loyaltyPoints.findUnique({
      where: { userId_tenantId: { userId, tenantId } }
    });

    if (!loyaltyPoints) {
      return { totalPoints: 0, usedPoints: 0, availablePoints: 0 };
    }

    return {
      totalPoints: loyaltyPoints.totalPoints,
      usedPoints: loyaltyPoints.usedPoints,
      availablePoints: loyaltyPoints.totalPoints - loyaltyPoints.usedPoints
    };
  }

  // পয়েন্ট হিস্ট্রি পান
  async getPointsHistory(userId: string) {
    const transactions = await this.prisma.pointsTransaction.findMany({
      where: { userId },
      orderBy: { createdAt: 'desc' },
      take: 20
    });

    return { transactions };
  }
}