import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class LoyaltyPointsService {
  constructor(private prisma: PrismaService) {}

  // প্রতিটি কেনাকাটায় পয়েন্ট যোগ করুন
  async addPointsOnPurchase(userId: string, orderTotal: number) {
    const points = Math.floor(orderTotal / 10); // প্রতি ১০ টাকায় ১ পয়েন্ট

    const loyaltyAccount = await this.prisma.loyaltyPoints.upsert({
      where: { userId },
      create: {
        userId,
        points,
        totalEarned: points
      },
      update: {
        points: { increment: points },
        totalEarned: { increment: points }
      }
    });

    return { success: true, pointsAdded: points, totalPoints: loyaltyAccount.points };
  }

  // পয়েন্ট রিডিম করুন
  async redeemPoints(userId: string, pointsToRedeem: number) {
    const loyalty = await this.prisma.loyaltyPoints.findUnique({
      where: { userId }
    });

    if (!loyalty || loyalty.points < pointsToRedeem) {
      throw new Error('Insufficient points');
    }

    // ডিসকাউন্ট তৈরি করুন
    const discountAmount = pointsToRedeem / 10; // ১০ পয়েন্ট = ১ টাকা

    const coupon = await this.prisma.coupon.create({
      data: {
        code: `POINTS${Math.random().toString(36).substring(7)}`,
        discountAmount,
        userId,
        expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000)
      }
    });

    // পয়েন্ট ব্যবহার করা হয়েছে হিসাবে চিহ্নিত করুন
    await this.prisma.loyaltyPoints.update({
      where: { userId },
      data: {
        points: { decrement: pointsToRedeem },
        totalRedeemed: { increment: pointsToRedeem }
      }
    });

    return { success: true, couponCode: coupon.code, discountAmount };
  }

  // VIP স্ট্যাটাস আপগ্রেড করুন (পয়েন্টের উপর ভিত্তি করে)
  async checkAndUpgradeVIPStatus(userId: string) {
    const loyalty = await this.prisma.loyaltyPoints.findUnique({
      where: { userId }
    });

    let vipLevel = 'bronze';
    if (loyalty!.totalEarned > 10000) {
      vipLevel = 'platinum';
    } else if (loyalty!.totalEarned > 5000) {
      vipLevel = 'gold';
    } else if (loyalty!.totalEarned > 2000) {
      vipLevel = 'silver';
    }

    await this.prisma.loyaltyPoints.update({
      where: { userId },
      data: { vipLevel }
    });

    return { vipLevel };
  }
}