import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { InjectQueue } from '@nestjs/bull';
import { Queue } from 'bull';

@Injectable()
export class CampaignService {
  private logger = new Logger('CampaignService');

  constructor(
    private prisma: PrismaService,
    @InjectQueue('email') private emailQueue: Queue
  ) {}

  // ক্যাম্পেইন তৈরি করুন
  async createCampaign(tenantId: string, input: {
    name: string;
    subject: string;
    htmlContent: string;
    segmentType: string;
    filters?: any;
  }) {
    const campaign = await this.prisma.emailCampaign.create({
      data: {
        tenantId,
        name: input.name,
        subject: input.subject,
        htmlContent: input.htmlContent,
        segmentType: input.segmentType,
        filters: input.filters,
        status: 'draft'
      }
    });

    // প্রাপকসংখ্যা গণনা করুন
    const recipientCount = await this.countRecipients(tenantId, input.segmentType, input.filters);
    await this.prisma.emailCampaign.update({
      where: { id: campaign.id },
      data: { recipientCount }
    });

    this.logger.log(`Campaign created: ${campaign.id}`);

    return { success: true, data: campaign };
  }

  // ক্যাম্পেইন সময়সূচী করুন
  async scheduleCampaign(campaignId: string, scheduledAt: Date) {
    const campaign = await this.prisma.emailCampaign.update({
      where: { id: campaignId },
      data: {
        status: 'scheduled',
        scheduledAt
      }
    });

    // কাজ কিউতে যোগ করুন
    await this.emailQueue.add(
      'send-campaign',
      { campaignId },
      { delay: scheduledAt.getTime() - Date.now() }
    );

    return { success: true, data: campaign };
  }

  // ক্যাম্পেইন পাঠান
  async sendCampaign(campaignId: string) {
    const campaign = await this.prisma.emailCampaign.findUnique({
      where: { id: campaignId }
    });

    if (!campaign) return { success: false, error: 'Campaign not found' };

    // সেগমেন্ট অনুযায়ী ইউজার পান
    const recipients = await this.getRecipients(
      campaign.tenantId,
      campaign.segmentType,
      campaign.filters
    );

    // ইমেইল কিউতে যোগ করুন
    for (const recipient of recipients) {
      await this.emailQueue.add('send-email', {
        to: recipient.email,
        subject: campaign.subject,
        htmlContent: campaign.htmlContent,
        campaignId
      });
    }

    // স্ট্যাটাস আপডেট করুন
    await this.prisma.emailCampaign.update({
      where: { id: campaignId },
      data: {
        status: 'sent',
        sentAt: new Date()
      }
    });

    return { success: true, sentCount: recipients.length };
  }

  private async countRecipients(
    tenantId: string,
    segmentType: string,
    filters?: any
  ): Promise<number> {
    const where: any = { tenantId };

    switch (segmentType) {
      case 'purchased':
        where.orders = { some: { status: 'DELIVERED' } };
        break;
      case 'inactive':
        // 30+ দিন কোন অর্ডার নেই
        const thirtyDaysAgo = new Date(Date.now() - 30 * 24 * 60 * 60 * 1000);
        where.orders = { none: { createdAt: { gt: thirtyDaysAgo } } };
        break;
      case 'custom':
        // কাস্টম ফিল্টার প্রয়োগ করুন
        if (filters?.minOrders) where.orders = { some: {} };
        break;
    }

    return await this.prisma.user.count({ where });
  }

  private async getRecipients(
    tenantId: string,
    segmentType: string,
    filters?: any
  ) {
    const where: any = { tenantId };

    if (segmentType === 'purchased') {
      where.orders = { some: { status: 'DELIVERED' } };
    }

    return await this.prisma.user.findMany({
      where,
      select: { email: true }
    });
  }

  // ক্যাম্পেইন অ্যানালিটিক্স পান
  async getCampaignAnalytics(campaignId: string) {
    const campaign = await this.prisma.emailCampaign.findUnique({
      where: { id: campaignId }
    });

    if (!campaign) return null;

    return {
      sent: campaign.recipientCount,
      openRate: campaign.openRate || 0,
      clickRate: campaign.clickRate || 0,
      conversionRate: campaign.conversionRate || 0,
      status: campaign.status
    };
  }
}