import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as nodemailer from 'nodemailer';

@Injectable()
export class CartRecoveryService {
  private logger = new Logger('CartRecoveryService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // ৩০ মিনিট পর কার্ট পরিত্যক্ত হলে ইমেইল পাঠান
  async sendAbandonedCartReminder() {
    const abandonedCarts = await this.prisma.abandonedCart.findMany({
      where: {
        recoveredAt: null,
        createdAt: {
          gte: new Date(Date.now() - 35 * 60 * 1000), // ৩৫ মিনিট
          lt: new Date(Date.now() - 30 * 60 * 1000) // ৩০ মিনিট
        }
      },
      include: { user: true }
    });

    for (const cart of abandonedCarts) {
      // বিশেষ ছাড় অফার তৈরি করুন
      const discountCode = await this.generateDiscountCode(
        cart.userId,
        10 // ১০% ছাড়
      );

      const emailContent = `
        <h2>আপনার কার্ট পরিত্যক্ত হয়েছে!</h2>
        <p>আপনার কার্টে ${cart.itemCount} আইটেম রয়েছে।</p>
        <p>এখনই কিনুন এবং ${discountCode} কোড দিয়ে ১০% ছাড় পান!</p>
        <a href="${process.env.APP_URL}/cart/${cart.id}">কার্ট দেখুন</a>
      `;

      await this.mailer.sendMail({
        from: process.env.STORE_EMAIL,
        to: cart.user.email,
        subject: '⏰ আপনার কার্ট অপেক্ষা করছে!',
        html: emailContent
      });

      this.logger.log(`Cart recovery email sent to ${cart.user.email}`);
    }
  }

  private async generateDiscountCode(
    userId: string,
    discountPercent: number
  ): Promise<string> {
    const code = `RECOVERY${Math.random().toString(36).substring(7).toUpperCase()}`;

    await this.prisma.coupon.create({
      data: {
        code,
        discountPercent,
        userId,
        expiresAt: new Date(Date.now() + 24 * 60 * 60 * 1000) // ২৪ ঘণ্টা
      }
    });

    return code;
  }
}