import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import Stripe from 'stripe';

@Injectable()
export class MarketplaceService {
  private logger = new Logger('MarketplaceService');
  private stripe = new Stripe(process.env.STRIPE_SECRET_KEY || '');

  constructor(private prisma: PrismaService) {}

  // মার্কেটপ্লেস প্রোডাক্ট পাবলিশ করুন
  async publishProduct(
    authorId: string,
    productData: any
  ) {
    const product = await this.prisma.marketplaceProduct.create({
      data: {
        ...productData,
        authorId,
        status: 'draft' // মডারেশনের জন্য
      }
    });

    return { success: true, productId: product.id };
  }

  // মার্কেটপ্লেস ব্রাউজ করুন
  async browseMarketplace(
    type?: 'theme' | 'plugin' | 'extension',
    category?: string,
    sortBy: 'popular' | 'new' | 'rating' = 'popular'
  ) {
    const products = await this.prisma.marketplaceProduct.findMany({
      where: {
        status: 'published',
        ...(type && { type }),
        // category filtering logic
      },
      orderBy: this.getSortOrder(sortBy),
      take: 20
    });

    return { products };
  }

  // প্রোডাক্ট সার্চ করুন
  async searchMarketplace(query: string) {
    const products = await this.prisma.marketplaceProduct.findMany({
      where: {
        status: 'published',
        OR: [
          { name: { contains: query, mode: 'insensitive' } },
          { description: { contains: query, mode: 'insensitive' } }
        ]
      }
    });

    return { products };
  }

  // প্রোডাক্ট কিনুন
  async purchaseProduct(
    tenantId: string,
    productId: string,
    paymentMethod: string
  ) {
    const product = await this.prisma.marketplaceProduct.findUnique({
      where: { id: productId }
    });

    if (!product) throw new Error('Product not found');

    // পেমেন্ট প্রসেস করুন
    const charge = await this.stripe.charges.create({
      amount: Number(product.price) * 100, // সেন্টে
      currency: product.currency.toLowerCase(),
      source: paymentMethod,
      description: `Purchase: ${product.name}`
    });

    // ক্রয় রেকর্ড করুন
    const purchase = await this.prisma.marketplacePurchase.create({
      data: {
        tenantId,
        productId,
        price: product.price
      }
    });

    // ডাউনলোড লিঙ্ক পাঠান
    // TODO: Send download link via email

    // ডাউনলোড কাউন্ট বাড়ান
    await this.prisma.marketplaceProduct.update({
      where: { id: productId },
      data: { downloads: { increment: 1 } }
    });

    return {
      success: true,
      purchaseId: purchase.id,
      downloadUrl: `https://marketplace.platform.com/download/${purchase.id}`
    };
  }

  // প্রোডাক্ট ইনস্টল করুন
  async installProduct(tenantId: string, productId: string) {
    // ক্রয় যাচাই করুন
    const purchase = await this.prisma.marketplacePurchase.findUnique({
      where: {
        tenantId_productId: { tenantId, productId }
      }
    });

    if (!purchase) throw new Error('Product not purchased');

    // প্রোডাক্ট ডাউনলোড করুন এবং ইনস্টল করুন
    // TODO: Install product logic

    return { success: true, installationId: Date.now() };
  }

  // প্রোডাক্ট রেটিং দিন
  async rateProduct(productId: string, rating: number, review?: string) {
    const product = await this.prisma.marketplaceProduct.findUnique({
      where: { id: productId }
    });

    if (!product) throw new Error('Product not found');

    // রেটিং ক্যালকুলেট করুন (সিম্পল এভারেজ)
    const newRating =
      (product.rating! * product.reviewCount + rating) /
      (product.reviewCount + 1);

    await this.prisma.marketplaceProduct.update({
      where: { id: productId },
      data: {
        rating: newRating,
        reviewCount: { increment: 1 }
      }
    });

    return { success: true };
  }

  private getSortOrder(sortBy: string) {
    switch (sortBy) {
      case 'new':
        return { createdAt: 'desc' };
      case 'rating':
        return { rating: 'desc' };
      case 'popular':
      default:
        return { downloads: 'desc' };
    }
  }
}