import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class MeteringService {
  private logger = new Logger('MeteringService');

  constructor(private prisma: PrismaService) {}

  // কী: প্রতিটি action ট্র্যাক করুন
  async logUsage(tenantId: string, event: string, metadata?: any) {
    await this.prisma.usageLog.create({
      data: {
        tenantId,
        event,
        metadata
      }
    });

    this.logger.debug(`Usage logged: ${tenantId} - ${event}`);
  }

  // অর্ডার তৈরি হলে ট্র্যাক করুন
  async trackOrderCreated(tenantId: string, orderId: string, amount: number) {
    await this.logUsage(tenantId, 'order_created', {
      orderId,
      amount
    });
  }

  // API কল ট্র্যাক করুন
  async trackApiCall(tenantId: string, endpoint: string, method: string) {
    await this.logUsage(tenantId, 'api_call', {
      endpoint,
      method
    });
  }

  // ফাইল/ইমেজ আপলোড ট্র্যাক করুন
  async trackStorageUsage(tenantId: string, fileSizeKb: number) {
    await this.logUsage(tenantId, 'file_uploaded', {
      sizeKb: fileSizeKb
    });
  }

  // মাসিক মেট্রিক্স ক্যালকুলেট করুন
  @Cron('0 0 1 * *') // প্রতি মাসের ১ তারিখ
  async calculateMonthlyMetrics() {
    const tenants = await this.prisma.tenant.findMany({
      where: { status: 'ACTIVE' }
    });

    for (const tenant of tenants) {
      await this.calculateTenantMetrics(tenant.id);
    }
  }

  private async calculateTenantMetrics(tenantId: string) {
    const startOfMonth = new Date();
    startOfMonth.setDate(1);
    startOfMonth.setHours(0, 0, 0, 0);

    const endOfMonth = new Date();
    endOfMonth.setMonth(endOfMonth.getMonth() + 1);
    endOfMonth.setDate(0);
    endOfMonth.setHours(23, 59, 59, 999);

    // অর্ডার গণনা করুন
    const orderCount = await this.prisma.order.count({
      where: {
        tenantId,
        createdAt: { gte: startOfMonth, lte: endOfMonth }
      }
    });

    // স্টোরেজ ব্যবহার গণনা করুন
    const storageUsage = await this.calculateStorageUsage(tenantId);

    // API কল গণনা করুন
    const apiCallCount = await this.prisma.usageLog.count({
      where: {
        tenantId,
        event: 'api_call',
        timestamp: { gte: startOfMonth, lte: endOfMonth }
      }
    });

    // সাবস্ক্রিপশন পরিকল্পনা পান
    const subscription = await this.prisma.subscription.findFirst({
      where: { tenantId }
    });

    const limits = this.getPlanLimits(subscription?.plan);

    // মেট্রিক্স সংরক্ষণ করুন
    await Promise.all([
      this.saveMetric(tenantId, 'orders', orderCount, limits.orders, startOfMonth, endOfMonth),
      this.saveMetric(tenantId, 'storage', storageUsage, limits.storage, startOfMonth, endOfMonth),
      this.saveMetric(tenantId, 'api_calls', apiCallCount, limits.apiCalls, startOfMonth, endOfMonth)
    ]);

    // চালান তৈরি করুন
    await this.generateInvoice(tenantId, startOfMonth, endOfMonth);
  }

  private async saveMetric(
    tenantId: string,
    metricType: string,
    usage: number,
    limit: number,
    startDate: Date,
    endDate: Date
  ) {
    const overage = Math.max(0, usage - limit);
    
    // Overage খরচ গণনা করুন
    const overageCost = this.calculateOverageCost(metricType, overage);

    await this.prisma.usageMetric.create({
      data: {
        tenantId,
        metricType,
        period: 'month',
        periodStart: startDate,
        periodEnd: endDate,
        usage,
        limit,
        overage,
        cost: overageCost
      }
    });

    this.logger.log(
      `Metric saved: ${tenantId} - ${metricType}: ${usage} (limit: ${limit}, overage: ${overage})`
    );
  }

  private getPlanLimits(plan: string | undefined) {
    const limits: any = {
      FREE: { orders: 100, storage: 1000, apiCalls: 1000 },
      STARTER: { orders: 1000, storage: 10000, apiCalls: 10000 },
      PROFESSIONAL: { orders: 10000, storage: 100000, apiCalls: 100000 },
      ENTERPRISE: { orders: -1, storage: -1, apiCalls: -1 } // Unlimited
    };

    return limits[plan] || limits.FREE;
  }

  private calculateOverageCost(metricType: string, overage: number): number {
    const costPerUnit: any = {
      orders: 0.5, // $0.50 per order
      storage: 0.1, // $0.10 per 1GB
      api_calls: 0.001 // $0.001 per 100 calls
    };

    return overage * (costPerUnit[metricType] || 0);
  }

  private async calculateStorageUsage(tenantId: string): Promise<number> {
    // TODO: Get actual file sizes from S3
    // For now, return mock data
    return 2500; // 2.5 GB in MB
  }

  // ব্যবহার রিপোর্ট পান
  async getUsageReport(tenantId: string, month?: string) {
    const now = new Date();
    const year = now.getFullYear();
    const currentMonth = String(now.getMonth() + 1).padStart(2, '0');
    const period = month || `${year}-${currentMonth}`;

    const metrics = await this.prisma.usageMetric.findMany({
      where: {
        tenantId,
        period: 'month',
        periodStart: {
          gte: new Date(`${period}-01`),
          lt: new Date(`${period}-32`)
        }
      }
    });

    const summary = {
      period,
      orders: metrics.find((m) => m.metricType === 'orders'),
      storage: metrics.find((m) => m.metricType === 'storage'),
      apiCalls: metrics.find((m) => m.metricType === 'api_calls'),
      totalOverageCost: metrics.reduce((sum, m) => sum + Number(m.cost), 0)
    };

    return { success: true, data: summary };
  }

  // চালান তৈরি করুন
  private async generateInvoice(tenantId: string, startDate: Date, endDate: Date) {
    // Get usage metrics
    const metrics = await this.prisma.usageMetric.findMany({
      where: { tenantId, periodStart: startDate }
    });

    // Get subscription cost
    const subscription = await this.prisma.subscription.findFirst({
      where: { tenantId }
    });

    const planCost = this.getPlanCost(subscription?.plan);

    // Calculate total
    const overageCost = metrics.reduce((sum, m) => sum + Number(m.cost), 0);
    const subtotal = planCost + overageCost;
    const tax = subtotal * 0.1;
    const total = subtotal + tax;

    // Create invoice
    const invoice = await this.prisma.billingInvoice.create({
      data: {
        tenantId,
        invoiceNumber: `INV-${tenantId.slice(0, 8)}-${startDate.getFullYear()}-${String(startDate.getMonth() + 1).padStart(2, '0')}`,
        status: 'draft',
        period: `${startDate.getFullYear()}-${String(startDate.getMonth() + 1).padStart(2, '0')}`,
        subscriptionCost: planCost,
        orderOverage: Number(metrics.find((m) => m.metricType === 'orders')?.cost) || 0,
        storageOverage: Number(metrics.find((m) => m.metricType === 'storage')?.cost) || 0,
        apiOverage: Number(metrics.find((m) => m.metricType === 'api_calls')?.cost) || 0,
        subtotal,
        tax,
        total,
        dueAt: new Date(endDate.getTime() + 30 * 24 * 60 * 60 * 1000) // 30 days payment terms
      }
    });

    this.logger.log(`Invoice created: ${invoice.invoiceNumber}`);

    // Generate PDF (TODO: Implement PDF generation)
    // await this.generateInvoicePDF(invoice);

    return invoice;
  }

  private getPlanCost(plan: string | undefined): number {
    const costs: any = {
      FREE: 0,
      STARTER: 29,
      PROFESSIONAL: 99,
      ENTERPRISE: 999
    };

    return costs[plan] || 0;
  }
}