import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import axios from 'axios';

@Injectable()
export class PerformanceMonitorService {
  private logger = new Logger('PerformanceMonitorService');

  constructor(private prisma: PrismaService) {}

  // প্রতি ঘণ্টায় পারফরম্যান্স চেক করুন
  @Cron('0 * * * *')
  async monitorPerformance() {
    const tenants = await this.prisma.tenant.findMany({
      where: { status: 'ACTIVE' }
    });

    for (const tenant of tenants) {
      await this.checkStorePerformance(tenant.id);
    }
  }

  private async checkStorePerformance(tenantId: string) {
    try {
      // PageSpeed Insights API ব্যবহার করুন (Google)
      const seoScore = await this.checkSEOScore(tenantId);
      const pageLoadTime = await this.checkPageLoadTime(tenantId);
      const uptime = await this.checkUptime(tenantId);

      await this.prisma.storePerformance.upsert({
        where: { tenantId },
        create: {
          tenantId,
          avgPageLoadTime: pageLoadTime,
          seoScore,
          currentUptime: uptime,
          thirtyDayUptime: uptime
        },
        update: {
          avgPageLoadTime: pageLoadTime,
          seoScore,
          currentUptime: uptime
        }
      });

      this.logger.log(`Performance metrics updated for ${tenantId}`);
    } catch (err) {
      this.logger.error('Performance monitoring failed:', err);
    }
  }

  private async checkSEOScore(tenantId: string): Promise<number> {
    // PageSpeed Insights API কল করুন
    try {
      const tenant = await this.prisma.tenant.findUnique({
        where: { id: tenantId }
      });

      const url = `${process.env.NEXT_PUBLIC_APP_URL}/${tenant?.subdomain}`;

      const response = await axios.get(
        `https://www.googleapis.com/pagespeedonline/v5/runPagespeed?url=${url}&key=${process.env.GOOGLE_API_KEY}`
      );

      return response.data.lighthouseResult?.categories?.seo?.score * 100 || 0;
    } catch (err) {
      return 0;
    }
  }

  private async checkPageLoadTime(tenantId: string): Promise<number> {
    // আপনার স্টোর থেকে পেজ লোড সময় মাপুন
    const startTime = Date.now();

    try {
      const tenant = await this.prisma.tenant.findUnique({
        where: { id: tenantId }
      });

      const url = `${process.env.NEXT_PUBLIC_APP_URL}/${tenant?.subdomain}`;
      await axios.get(url, { timeout: 10000 });

      return Date.now() - startTime; // milliseconds
    } catch (err) {
      return 5000; // Default timeout
    }
  }

  private async checkUptime(tenantId: string): Promise<number> {
    // Uptime প্রতিশত ক্যালকুলেট করুন
    // এটি একটি সরলীকৃত উপায় - প্রকৃত সিস্টেম আরও জটিল
    return 99.9;
  }

  // পারফরম্যান্স রিপোর্ট পান
  async getPerformanceReport(tenantId: string) {
    const performance = await this.prisma.storePerformance.findUnique({
      where: { tenantId }
    });

    return {
      success: true,
      data: performance
    };
  }
}