import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import axios from 'axios';

@Injectable()
export class UptimeMonitorService {
  private logger = new Logger('UptimeMonitorService');

  constructor(private prisma: PrismaService) {}

  // প্রতি মিনিটে আপটাইম চেক করুন
  @Cron('* * * * *')
  async checkUptime() {
    const tenants = await this.prisma.tenant.findMany();

    for (const tenant of tenants) {
      try {
        const startTime = Date.now();

        // হোমপেজে রিকোয়েস্ট পাঠান
        await axios.get(`${process.env.APP_URL}/${tenant.id}`, {
          timeout: 5000
        });

        const responseTime = Date.now() - startTime;

        // মেট্রিক্স রেকর্ড করুন
        await this.prisma.uptimeMetrics.create({
          data: {
            tenantId: tenant.id,
            uptime: 100, // সফল রেসপন্স
            avgResponseTime: responseTime,
            avgPageLoadTime: responseTime,
            errors: 0
          }
        });
      } catch (err) {
        this.logger.warn(`Uptime check failed for tenant ${tenant.id}`);

        // ডাউনটাইম রেকর্ড করুন
        await this.prisma.uptimeMetrics.create({
          data: {
            tenantId: tenant.id,
            uptime: 0,
            avgResponseTime: 5000,
            avgPageLoadTime: 5000,
            errors: 1
          }
        });
      }
    }
  }

  // আপটাইম রিপোর্ট পান
  async getUptimeReport(tenantId: string, days: number = 30) {
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    const metrics = await this.prisma.uptimeMetrics.findMany({
      where: {
        tenantId,
        recordedAt: { gte: startDate }
      },
      orderBy: { recordedAt: 'asc' }
    });

    const totalChecks = metrics.length;
    const successfulChecks = metrics.filter((m) => m.uptime === 100).length;
    const avgUptime = (successfulChecks / totalChecks) * 100;

    const avgResponseTime =
      metrics.reduce((sum, m) => sum + m.avgResponseTime, 0) / totalChecks;

    const avgPageLoadTime =
      metrics.reduce((sum, m) => sum + m.avgPageLoadTime, 0) / totalChecks;

    return {
      uptime: avgUptime,
      avgResponseTime,
      avgPageLoadTime,
      totalChecks,
      successfulChecks
    };
  }
}