import { Injectable, Logger } from '@nestjs/common';
import axios from 'axios';
import * as nodemailer from 'nodemailer';

@Injectable()
export class MultiChannelNotificationService {
  private logger = new Logger('MultiChannelNotificationService');
  private emailTransporter = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  // মাল্টি-চ্যানেল অর্ডার নোটিফিকেশন
  async notifyOrderStatus(
    orderId: string,
    status: string,
    customerEmail: string,
    customerPhone: string
  ) {
    const messages = {
      PENDING: 'Your order has been placed successfully!',
      PROCESSING: 'Your order is being processed.',
      SHIPPED: 'Your order has been shipped!',
      DELIVERED: 'Your order has been delivered!'
    };

    const message = messages[status as keyof typeof messages] || 'Order status updated';

    // ইমেইল পাঠান
    await this.sendEmail(customerEmail, `Order Status: ${status}`, message);

    // SMS পাঠান
    await this.sendSMS(customerPhone, message);

    // পুশ নোটিফিকেশন পাঠান
    // await this.sendPushNotification(orderId, message);

    this.logger.log(`Notifications sent for order ${orderId}`);
  }

  private async sendEmail(to: string, subject: string, text: string) {
    try {
      await this.emailTransporter.sendMail({
        from: 'noreply@store.com',
        to,
        subject,
        html: `<p>${text}</p>`
      });
      this.logger.log(`Email sent to ${to}`);
    } catch (err) {
      this.logger.error('Email send failed:', err);
    }
  }

  private async sendSMS(phone: string, message: string) {
    try {
      // Twilio বা অন্য SMS গেটওয়ে ব্যবহার করুন
      await axios.post('https://api.twilio.com/2010-04-01/Accounts/SMS', {
        To: phone,
        From: process.env.TWILIO_PHONE,
        Body: message
      });
      this.logger.log(`SMS sent to ${phone}`);
    } catch (err) {
      this.logger.error('SMS send failed:', err);
    }
  }
}