import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class PriceNotificationService {
  private logger = new Logger('PriceNotificationService');

  constructor(private prisma: PrismaService) {}

  // প্রাইস চেঞ্জ নোটিফাই করুন
  async notifyPriceChange(
    productId: string,
    oldPrice: number,
    newPrice: number,
    adminTenantIds: string[]
  ) {
    try {
      // নোটিফিকেশন সংরক্ষণ করুন
      await this.prisma.priceChangeNotification.create({
        data: {
          productId,
          oldPrice,
          newPrice,
          notifiedAdmins: adminTenantIds
        }
      });

      // ইমেইল পাঠান
      for (const adminTenantId of adminTenantIds) {
        await this.sendEmailNotification(productId, oldPrice, newPrice, adminTenantId);
      }

      this.logger.log(`Price change notification sent for product ${productId}`);

      return { success: true };
    } catch (err) {
      this.logger.error('Notification failed:', err);
      throw err;
    }
  }

  private async sendEmailNotification(
    productId: string,
    oldPrice: number,
    newPrice: number,
    adminTenantId: string
  ) {
    // ইমেইল পাঠান
    // TODO: Implement email service integration
    const percent = Math.round(((newPrice - oldPrice) / oldPrice) * 100);
    const direction = newPrice > oldPrice ? 'increased' : 'decreased';

    this.logger.log(
      `Email: Price ${direction} from $${oldPrice} to $${newPrice} (${percent}%)`
    );
  }
}