import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as webpush from 'web-push';

@Injectable()
export class PushNotificationService {
  private logger = new Logger('PushNotificationService');

  constructor(private prisma: PrismaService) {
    webpush.setVapidDetails(
      process.env.VAPID_SUBJECT || '',
      process.env.VAPID_PUBLIC_KEY || '',
      process.env.VAPID_PRIVATE_KEY || ''
    );
  }

  // পুশ সাবস্ক্রিপশন সংরক্ষণ করুন
  async subscribeToNotifications(userId: string, subscription: any) {
    const sub = await this.prisma.pushSubscription.create({
      data: {
        userId,
        endpoint: subscription.endpoint,
        auth: subscription.keys.auth,
        p256dh: subscription.keys.p256dh
      }
    });

    return { success: true };
  }

  // পুশ নোটিফিকেশন পাঠান
  async sendNotification(userId: string, title: string, message: string) {
    const subscriptions = await this.prisma.pushSubscription.findMany({
      where: { userId, isActive: true }
    });

    for (const sub of subscriptions) {
      try {
        await webpush.sendNotification(
          {
            endpoint: sub.endpoint,
            keys: {
              auth: sub.auth,
              p256dh: sub.p256dh
            }
          },
          JSON.stringify({
            title,
            body: message,
            icon: '/logo-192.png'
          })
        );

        this.logger.log(`Push notification sent to ${userId}`);
      } catch (err) {
        this.logger.error('Push notification failed:', err);

        // সাবস্ক্রিপশন ইনভ্যালিড হলে ডিসঅ্যাক্টিভেট করুন
        if (err instanceof Error && err.message.includes('410')) {
          await this.prisma.pushSubscription.update({
            where: { id: sub.id },
            data: { isActive: false }
          });
        }
      }
    }
  }

  // অর্ডার আপডেট নোটিফিকেশন
  async notifyOrderUpdate(userId: string, orderId: string, status: string) {
    const statusMessages: any = {
      PROCESSING: 'Your order is being prepared',
      SHIPPED: 'Your order has been shipped',
      DELIVERED: 'Your order has been delivered'
    };

    await this.sendNotification(
      userId,
      'Order Update',
      statusMessages[status] || 'Your order has been updated'
    );
  }
}