import { Injectable, Logger } from '@nestjs/common';
import * as sharp from 'sharp';
import * as AWS from 'aws-sdk';

@Injectable()
export class ImageOptimizationService {
  private logger = new Logger('ImageOptimizationService');
  private s3 = new AWS.S3();

  // ইমেজ কম্প্রেস করুন (WebP + AVIF)
  async optimizeImage(imageUrl: string, tenantId: string) {
    try {
      // ইমেজ ডাউনলোড করুন
      const response = await fetch(imageUrl);
      const buffer = await response.buffer();

      // WebP তৈরি করুন
      const webpBuffer = await sharp(buffer)
        .webp({ quality: 80 })
        .toBuffer();

      // AVIF তৈরি করুন
      const avifBuffer = await sharp(buffer)
        .avif({ quality: 75 })
        .toBuffer();

      // S3-এ আপলোড করুন
      const baseName = imageUrl.split('/').pop()?.split('.')[0];

      const webpUrl = await this.uploadToS3(
        tenantId,
        `${baseName}.webp`,
        webpBuffer,
        'image/webp'
      );

      const avifUrl = await this.uploadToS3(
        tenantId,
        `${baseName}.avif`,
        avifBuffer,
        'image/avif'
      );

      this.logger.log(`Image optimized: ${imageUrl}`);

      return {
        webp: webpUrl,
        avif: avifUrl,
        original: imageUrl,
        originalSize: buffer.length,
        webpSize: webpBuffer.length,
        avifSize: avifBuffer.length
      };
    } catch (err) {
      this.logger.error('Image optimization failed:', err);
      throw err;
    }
  }

  private async uploadToS3(
    tenantId: string,
    filename: string,
    buffer: Buffer,
    contentType: string
  ): Promise<string> {
    const key = `optimized-images/${tenantId}/${filename}`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: buffer,
        ContentType: contentType
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }
}