import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class OrderTrackingService {
  constructor(private prisma: PrismaService) {}

  // অর্ডার ট্র্যাকিং পান
  async getOrderTracking(orderId: string) {
    const order = await this.prisma.order.findUnique({
      where: { id: orderId },
      include: {
        items: { include: { product: true } },
        shippingAddress: true
      }
    });

    if (!order) throw new Error('Order not found');

    // স্ট্যাটাস টাইমলাইন তৈরি করুন
    const timeline = this.buildStatusTimeline(order);

    return {
      success: true,
      data: {
        orderNumber: order.orderNumber,
        status: order.status,
        total: order.total,
        items: order.items,
        timeline,
        estimatedDelivery: this.calculateEstimatedDelivery(order),
        shippingAddress: order.shippingAddress,
        trackingNumber: order.trackingNumber,
        courier: order.courier
      }
    };
  }

  // স্ট্যাটাস টাইমলাইন বিল্ড করুন
  private buildStatusTimeline(order: any) {
    const statuses = ['PENDING', 'PROCESSING', 'SHIPPED', 'DELIVERED'];
    const timeline = [];

    for (const status of statuses) {
      const isCompleted = statuses.indexOf(status) <= statuses.indexOf(order.status);

      timeline.push({
        status,
        label: this.getStatusLabel(status),
        completed: isCompleted,
        timestamp: this.getStatusTimestamp(status, order),
        icon: this.getStatusIcon(status)
      });
    }

    return timeline;
  }

  private getStatusLabel(status: string): string {
    const labels: any = {
      PENDING: 'Order Placed',
      PROCESSING: 'Processing',
      SHIPPED: 'Shipped',
      DELIVERED: 'Delivered'
    };
    return labels[status] || status;
  }

  private getStatusIcon(status: string): string {
    const icons: any = {
      PENDING: '📦',
      PROCESSING: '⚙️',
      SHIPPED: '🚚',
      DELIVERED: '✅'
    };
    return icons[status] || '📍';
  }

  private getStatusTimestamp(status: string, order: any): Date | null {
    // উপযুক্ত টাইমস্ট্যাম্প রিটার্ন করুন
    if (status === 'PENDING') return order.createdAt;
    if (status === order.status) return new Date();
    return null;
  }

  private calculateEstimatedDelivery(order: any): Date {
    // ডেলিভারি তারিখ ক্যালকুলেট করুন
    const deliveryDate = new Date(order.createdAt);
    deliveryDate.setDate(deliveryDate.getDate() + 7); // 7 দিনের ডিফল্ট
    return deliveryDate;
  }
}