import {
  Controller,
  Post,
  Get,
  Put,
  Body,
  Param,
  Query,
  UseGuards,
  Req
} from '@nestjs/common';
import { ApiTags, ApiBearerAuth } from '@nestjs/swagger';
import { OrderService } from './order.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';

@ApiTags('Orders')
@Controller('orders')
@UseGuards(JwtAuthGuard)
export class OrderController {
  constructor(private orderService: OrderService) {}

  @Post()
  @ApiBearerAuth()
  async create(@Req() req: any, @Body() dto: any) {
    return await this.orderService.createOrder(req.user.tenantId, req.user.sub, dto);
  }

  @Get()
  @ApiBearerAuth()
  async list(
    @Req() req: any,
    @Query('skip') skip = 0,
    @Query('take') take = 20,
    @Query('status') status?: string
  ) {
    return await this.orderService.listOrders(req.user.tenantId, skip, take, { status });
  }

  @Get(':id')
  @ApiBearerAuth()
  async findOne(@Req() req: any, @Param('id') id: string) {
    return await this.orderService.getOrder(req.user.tenantId, id);
  }

  @Put(':id/status')
  @ApiBearerAuth()
  async updateStatus(@Req() req: any, @Param('id') id: string, @Body() body: { status: string }) {
    return await this.orderService.updateOrderStatus(req.user.tenantId, id, body.status);
  }

  @Post(':id/payment-confirm')
  @ApiBearerAuth()
  async confirmPayment(
    @Req() req: any,
    @Param('id') id: string,
    @Body() body: { transactionId: string }
  ) {
    return await this.orderService.markPaymentSuccess(req.user.tenantId, id, body.transactionId);
  }

  @Delete(':id')
  @ApiBearerAuth()
  async cancel(@Req() req: any, @Param('id') id: string) {
    return await this.orderService.cancelOrder(req.user.tenantId, id);
  }
}