import { Injectable, BadRequestException, NotFoundException, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { InjectQueue } from '@nestjs/bull';
import { Queue } from 'bull';

@Injectable()
export class OrderService {
  private logger = new Logger('OrderService');

  constructor(
    private prisma: PrismaService,
    @InjectQueue('email') private emailQueue: Queue,
    @InjectQueue('webhooks') private webhookQueue: Queue
  ) {}

  async createOrder(tenantId: string, userId: string, dto: any) {
    const user = await this.prisma.user.findFirst({
      where: { id: userId, tenantId }
    });

    if (!user) {
      throw new NotFoundException('User not found');
    }

    // Validate cart items & calculate total
    const cartValidation = await this.validateAndPriceCart(tenantId, dto.items);

    // Generate order number
    const orderNumber = `ORD-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;

    // Create order
    const order = await this.prisma.order.create({
      data: {
        tenantId,
        orderNumber,
        userId,
        subtotal: cartValidation.subtotal,
        tax: cartValidation.tax,
        shipping: cartValidation.shipping,
        total: cartValidation.total,
        status: 'PENDING',
        paymentStatus: 'pending',
        shippingAddress: JSON.stringify(dto.shippingAddress),
        billingAddress: JSON.stringify(dto.billingAddress || dto.shippingAddress),
        notes: dto.notes,
        items: {
          create: cartValidation.items.map((item) => ({
            productId: item.id,
            quantity: item.quantity,
            price: item.price,
            total: item.subtotal,
            variantData: item.variantData
          }))
        }
      },
      include: { items: { include: { product: true } } }
    });

    // Reserve stock
    for (const item of cartValidation.items) {
      await this.prisma.product.update({
        where: { id: item.id },
        data: { stock: { decrement: item.quantity } }
      });
    }

    this.logger.log(`Order created: ${order.orderNumber}`);

    // Queue confirmation email
    await this.emailQueue.add('send-order-confirmation', {
      email: user.email,
      orderNumber: order.orderNumber,
      totalAmount: order.total,
      tenantHost: (await this.prisma.tenant.findUnique({ where: { id: tenantId } }))?.host
    });

    // Queue webhook
    await this.webhookQueue.add('trigger', {
      tenantId,
      event: 'order.created',
      data: order
    });

    return {
      success: true,
      data: order
    };
  }

  async getOrder(tenantId: string, orderId: string) {
    const order = await this.prisma.order.findFirst({
      where: { id: orderId, tenantId },
      include: { items: { include: { product: true } }, user: true }
    });

    if (!order) {
      throw new NotFoundException('Order not found');
    }

    return { data: order };
  }

  async listOrders(tenantId: string, skip = 0, take = 20, filters?: any) {
    const where: any = { tenantId };

    if (filters?.status) where.status = filters.status;
    if (filters?.paymentStatus) where.paymentStatus = filters.paymentStatus;
    if (filters?.userId) where.userId = filters.userId;

    const [orders, total] = await Promise.all([
      this.prisma.order.findMany({
        where,
        skip,
        take,
        include: { user: true },
        orderBy: { createdAt: 'desc' }
      }),
      this.prisma.order.count({ where })
    ]);

    return {
      items: orders,
      total,
      page: Math.floor(skip / take) + 1,
      limit: take
    };
  }

  async updateOrderStatus(tenantId: string, orderId: string, status: string) {
    const order = await this.prisma.order.findFirst({
      where: { id: orderId, tenantId }
    });

    if (!order) {
      throw new NotFoundException('Order not found');
    }

    const updated = await this.prisma.order.update({
      where: { id: orderId },
      data: {
        status,
        updatedAt: new Date()
      },
      include: { items: true, user: true }
    });

    // Queue status update email
    await this.emailQueue.add('send-order-status-update', {
      email: updated.user.email,
      orderNumber: updated.orderNumber,
      status,
      tenantHost: (await this.prisma.tenant.findUnique({ where: { id: tenantId } }))?.host
    });

    // Queue webhook
    await this.webhookQueue.add('trigger', {
      tenantId,
      event: 'order.status_updated',
      data: updated
    });

    return { success: true, data: updated };
  }

  async markPaymentSuccess(tenantId: string, orderId: string, transactionId: string) {
    const order = await this.prisma.order.findFirst({
      where: { id: orderId, tenantId }
    });

    if (!order) {
      throw new NotFoundException('Order not found');
    }

    const updated = await this.prisma.order.update({
      where: { id: orderId },
      data: {
        paymentStatus: 'completed',
        status: 'CONFIRMED',
        transactionId,
        updatedAt: new Date()
      },
      include: { items: true, user: true }
    });

    // Queue webhook
    await this.webhookQueue.add('trigger', {
      tenantId,
      event: 'order.payment_confirmed',
      data: updated
    });

    return { success: true, data: updated };
  }

  async cancelOrder(tenantId: string, orderId: string) {
    const order = await this.prisma.order.findFirst({
      where: { id: orderId, tenantId },
      include: { items: true }
    });

    if (!order) {
      throw new NotFoundException('Order not found');
    }

    // Release stock
    for (const item of order.items) {
      await this.prisma.product.update({
        where: { id: item.productId },
        data: { stock: { increment: item.quantity } }
      });
    }

    const updated = await this.prisma.order.update({
      where: { id: orderId },
      data: {
        status: 'CANCELLED',
        updatedAt: new Date()
      }
    });

    // Queue webhook
    await this.webhookQueue.add('trigger', {
      tenantId,
      event: 'order.cancelled',
      data: updated
    });

    return { success: true, data: updated };
  }

  private async validateAndPriceCart(tenantId: string, items: any[]) {
    const validItems = [];

    for (const item of items) {
      const product = await this.prisma.product.findFirst({
        where: { id: item.productId, tenantId }
      });

      if (!product) {
        throw new BadRequestException(`Product ${item.productId} not found`);
      }

      if (product.stock < item.quantity) {
        throw new BadRequestException(`Insufficient stock for ${product.title}`);
      }

      validItems.push({
        id: product.id,
        sku: product.sku,
        title: product.title,
        price: product.price,
        quantity: item.quantity,
        subtotal: product.price * item.quantity,
        variantData: item.variantData
      });
    }

    const subtotal = validItems.reduce((sum, item) => sum + item.subtotal, 0);
    const tax = subtotal * 0.1; // 10% tax
    const shipping = subtotal > 100 ? 0 : 10; // Free shipping over $100

    return {
      items: validItems,
      subtotal,
      tax,
      shipping,
      total: subtotal + tax + shipping
    };
  }
}