import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class WholesaleOrderService {
  private logger = new Logger('WholesaleOrderService');

  constructor(private prisma: PrismaService) {}

  // হোলসেলার অর্ডার তৈরি করুন (স্টোর থেকে)
  async createWholesaleOrder(
    adminTenantId: string,
    wholesalerTenantId: string,
    items: Array<{
      productId: string;
      quantity: number;
    }>
  ) {
    let totalAmount = 0;
    let totalQty = 0;

    // আইটেম বিস্তারিত পান এবং মূল্য ক্যালকুলেট করুন
    const orderItems: any[] = [];

    for (const item of items) {
      const product = await this.prisma.product.findUnique({
        where: { id: item.productId }
      });

      if (!product) throw new Error(`Product not found: ${item.productId}`);

      const itemTotal = Number(product.price) * item.quantity;
      totalAmount += itemTotal;
      totalQty += item.quantity;

      orderItems.push({
        productId: item.productId,
        quantity: item.quantity,
        unitPrice: product.price,
        totalPrice: itemTotal
      });
    }

    // অর্ডার তৈরি করুন
    const order = await this.prisma.order.create({
      data: {
        tenantId: adminTenantId,
        userId: 'wholesaler-api', // System order
        orderNumber: `WS-${Date.now()}`,
        status: 'PENDING',
        total: totalAmount,
        tax: 0,
        items: {
          create: orderItems.map((item) => ({
            ...item,
            wholesalerId: wholesalerTenantId
          }))
        }
      }
    });

    this.logger.log(`Wholesale order created: ${order.id}`);

    return {
      success: true,
      data: {
        orderId: order.id,
        orderNumber: order.orderNumber,
        totalQty,
        totalAmount
      }
    };
  }

  // হোলসেলার অর্ডার সার্চ করুন
  async getWholesaleOrders(
    wholesalerTenantId: string,
    status?: string,
    limit = 50
  ) {
    const where: any = {
      items: {
        some: {
          wholesalerId: wholesalerTenantId
        }
      }
    };

    if (status) where.status = status;

    return await this.prisma.order.findMany({
      where,
      take: limit,
      orderBy: { createdAt: 'desc' },
      include: { items: true }
    });
  }
}