import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CountryPaymentService {
  private logger = new Logger('CountryPaymentService');

  constructor(private prisma: PrismaService) {}

  // দেশের পেমেন্ট গেটওয়ে পান
  async getCountryPaymentGateways(countryCode: string) {
    const gateways = await this.prisma.countryPaymentGateway.findMany({
      where: {
        countryCode,
        isEnabled: true
      },
      orderBy: { priority: 'asc' }
    });

    return gateways;
  }

  // পেমেন্ট গেটওয়ে কনফিগার করুন
  async configurePaymentGateway(
    countryCode: string,
    gatewayName: string,
    config: any,
    priority: number = 0
  ) {
    const gateway = await this.prisma.countryPaymentGateway.upsert({
      where: {
        countryCode_gatewayName: { countryCode, gatewayName }
      },
      create: {
        countryCode,
        gatewayName,
        config,
        priority,
        isEnabled: true
      },
      update: {
        config,
        priority,
        isEnabled: true
      }
    });

    this.logger.log(`Payment gateway configured: ${gatewayName} in ${countryCode}`);

    return { success: true, data: gateway };
  }

  // দেশের জন্য পেমেন্ট প্রসেস করুন
  async processCountryPayment(
    countryCode: string,
    orderId: string,
    amount: number,
    currency: string
  ) {
    // উপযুক্ত গেটওয়ে খুঁজুন
    const gateway = await this.prisma.countryPaymentGateway.findFirst({
      where: {
        countryCode,
        isEnabled: true
      },
      orderBy: { priority: 'asc' }
    });

    if (!gateway) {
      throw new Error(`No payment gateway configured for ${countryCode}`);
    }

    // গেটওয়ে-নির্দিষ্ট পেমেন্ট প্রসেসিং
    return await this.processViaGateway(gateway.gatewayName, {
      orderId,
      amount,
      currency,
      config: gateway.config
    });
  }

  private async processViaGateway(
    gatewayName: string,
    paymentData: any
  ) {
    switch (gatewayName) {
      case 'bkash':
        return await this.processBkashPayment(paymentData);
      case 'nagad':
        return await this.processNagadPayment(paymentData);
      case 'sslcommerz':
        return await this.processSSLComerzPayment(paymentData);
      case 'razorpay':
        return await this.processRazorpayPayment(paymentData);
      default:
        throw new Error(`Unknown gateway: ${gatewayName}`);
    }
  }

  private async processBkashPayment(data: any) {
    // bKash API integration
    this.logger.log(`Processing bKash payment: ${data.orderId}`);
    return { success: true, gateway: 'bkash' };
  }

  private async processNagadPayment(data: any) {
    // Nagad API integration
    this.logger.log(`Processing Nagad payment: ${data.orderId}`);
    return { success: true, gateway: 'nagad' };
  }

  private async processSSLComerzPayment(data: any) {
    // SSLCommerz API integration
    this.logger.log(`Processing SSLCommerz payment: ${data.orderId}`);
    return { success: true, gateway: 'sslcommerz' };
  }

  private async processRazorpayPayment(data: any) {
    // Razorpay API integration
    this.logger.log(`Processing Razorpay payment: ${data.orderId}`);
    return { success: true, gateway: 'razorpay' };
  }
}