import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class PaymentAnalyticsService {
  private logger = new Logger('PaymentAnalyticsService');

  constructor(private prisma: PrismaService) {}

  async getPaymentAnalytics(tenantId: string, startDate: Date, endDate: Date) {
    const transactions = await this.prisma.paymentTransaction.findMany({
      where: {
        order: { tenantId },
        createdAt: { gte: startDate, lte: endDate }
      },
      include: { order: true }
    });

    // Group by gateway
    const byGateway: any = {};
    transactions.forEach((tx) => {
      if (!byGateway[tx.gatewayName]) {
        byGateway[tx.gatewayName] = {
          count: 0,
          total: 0,
          successful: 0,
          failed: 0
        };
      }
      byGateway[tx.gatewayName].count += 1;
      byGateway[tx.gatewayName].total += Number(tx.amount);
      if (tx.status === 'success') byGateway[tx.gatewayName].successful += 1;
      if (tx.status === 'failed') byGateway[tx.gatewayName].failed += 1;
    });

    const totalAmount = transactions.reduce((sum, tx) => sum + Number(tx.amount), 0);
    const successRate = transactions.length > 0
      ? (transactions.filter((tx) => tx.status === 'success').length / transactions.length * 100)
      : 0;

    return {
      data: {
        period: { start: startDate, end: endDate },
        totalTransactions: transactions.length,
        totalAmount,
        successRate: successRate.toFixed(2) + '%',
        byGateway
      }
    };
  }

  async reconcilePayments(tenantId: string) {
    // Compare database records with gateway records
    // Identify discrepancies and flag for resolution
    this.logger.log(`Reconciling payments for tenant ${tenantId}`);

    return {
      success: true,
      message: 'Reconciliation complete'
    };
  }

  async retryFailedPayments(tenantId: string) {
    // Find failed transactions older than 5 minutes
    // Attempt to verify/complete them

    return {
      success: true,
      message: 'Retry process started'
    };
  }
}