import { Injectable, BadRequestException, Logger } from '@nestjs/common';
import Stripe from 'stripe';
import axios from 'axios';

export interface PaymentGatewayConfig {
  name: string;
  apiKey?: string;
  apiSecret?: string;
  merchantId?: string;
  businessId?: string;
  storeId?: string;
  password?: string;
  endpoint?: string;
  currency?: string;
  region?: string;
  enabled: boolean;
}

export interface PaymentSession {
  gatewayName: string;
  sessionId: string;
  checkoutUrl?: string;
  redirectUrl?: string;
  amount: number;
  currency: string;
  orderId: string;
  status: string;
}

@Injectable()
export class PaymentGatewayService {
  private logger = new Logger('PaymentGatewayService');
  private stripe: Stripe;

  // Gateway configurations (loaded from env/database)
  private gateways: Map<string, PaymentGatewayConfig> = new Map();

  constructor() {
    this.initializeGateways();
  }

  private initializeGateways() {
    // Americas
    this.registerGateway({
      name: 'Stripe',
      apiKey: process.env.STRIPE_SECRET_KEY,
      enabled: !!process.env.STRIPE_SECRET_KEY,
      currency: 'USD',
      region: 'global'
    });

    this.registerGateway({
      name: 'PayPal',
      apiKey: process.env.PAYPAL_CLIENT_ID,
      apiSecret: process.env.PAYPAL_SECRET,
      enabled: !!process.env.PAYPAL_CLIENT_ID,
      currency: 'USD',
      region: 'global'
    });

    // Asia-Pacific
    this.registerGateway({
      name: 'Razorpay',
      apiKey: process.env.RAZORPAY_KEY_ID,
      apiSecret: process.env.RAZORPAY_KEY_SECRET,
      enabled: !!process.env.RAZORPAY_KEY_ID,
      currency: 'INR',
      region: 'IN'
    });

    this.registerGateway({
      name: 'SSLCommerz',
      storeId: process.env.SSLCOMMERZ_STORE_ID,
      password: process.env.SSLCOMMERZ_PASSWORD,
      enabled: !!process.env.SSLCOMMERZ_STORE_ID,
      currency: 'BDT',
      region: 'BD'
    });

    this.registerGateway({
      name: 'bKash PG',
      apiKey: process.env.BKASH_APP_KEY,
      apiSecret: process.env.BKASH_APP_SECRET,
      enabled: !!process.env.BKASH_APP_KEY,
      currency: 'BDT',
      region: 'BD'
    });

    this.registerGateway({
      name: 'Nagad',
      merchantId: process.env.NAGAD_MERCHANT_ID,
      apiKey: process.env.NAGAD_API_KEY,
      enabled: !!process.env.NAGAD_API_KEY,
      currency: 'BDT',
      region: 'BD'
    });

    this.registerGateway({
      name: 'JazzCash',
      merchantId: process.env.JAZZCASH_MERCHANT_ID,
      apiKey: process.env.JAZZCASH_API_KEY,
      enabled: !!process.env.JAZZCASH_API_KEY,
      currency: 'PKR',
      region: 'PK'
    });

    this.registerGateway({
      name: 'Easypaisa',
      merchantId: process.env.EASYPAISA_MERCHANT_ID,
      apiKey: process.env.EASYPAISA_API_KEY,
      enabled: !!process.env.EASYPAISA_API_KEY,
      currency: 'PKR',
      region: 'PK'
    });

    this.registerGateway({
      name: 'Alipay',
      merchantId: process.env.ALIPAY_MERCHANT_ID,
      apiKey: process.env.ALIPAY_API_KEY,
      enabled: !!process.env.ALIPAY_API_KEY,
      currency: 'CNY',
      region: 'CN'
    });

    this.registerGateway({
      name: 'WeChat Pay',
      merchantId: process.env.WECHAT_MERCHANT_ID,
      apiKey: process.env.WECHAT_API_KEY,
      enabled: !!process.env.WECHAT_API_KEY,
      currency: 'CNY',
      region: 'CN'
    });

    this.registerGateway({
      name: 'PayMongo',
      apiKey: process.env.PAYMONGO_SECRET_KEY,
      enabled: !!process.env.PAYMONGO_SECRET_KEY,
      currency: 'PHP',
      region: 'PH'
    });

    this.registerGateway({
      name: 'GCash API',
      merchantId: process.env.GCASH_MERCHANT_ID,
      apiKey: process.env.GCASH_API_KEY,
      enabled: !!process.env.GCASH_API_KEY,
      currency: 'PHP',
      region: 'PH'
    });

    this.registerGateway({
      name: 'Xendit',
      apiKey: process.env.XENDIT_API_KEY,
      enabled: !!process.env.XENDIT_API_KEY,
      currency: 'IDR',
      region: 'ID'
    });

    this.registerGateway({
      name: 'Khalti',
      apiKey: process.env.KHALTI_PUBLIC_KEY,
      apiSecret: process.env.KHALTI_SECRET_KEY,
      enabled: !!process.env.KHALTI_PUBLIC_KEY,
      currency: 'NPR',
      region: 'NP'
    });

    this.registerGateway({
      name: 'eSewa',
      merchantId: process.env.ESEWA_MERCHANT_CODE,
      enabled: !!process.env.ESEWA_MERCHANT_CODE,
      currency: 'NPR',
      region: 'NP'
    });

    this.registerGateway({
      name: 'Paytm',
      merchantId: process.env.PAYTM_MERCHANT_ID,
      apiKey: process.env.PAYTM_MERCHANT_KEY,
      enabled: !!process.env.PAYTM_MERCHANT_ID,
      currency: 'INR',
      region: 'IN'
    });

    this.registerGateway({
      name: 'PayHere',
      businessId: process.env.PAYHERE_BUSINESS_ID,
      apiKey: process.env.PAYHERE_API_KEY,
      enabled: !!process.env.PAYHERE_BUSINESS_ID,
      currency: 'LKR',
      region: 'LK'
    });

    this.registerGateway({
      name: 'Thawani Pay',
      apiKey: process.env.THAWANI_API_KEY,
      enabled: !!process.env.THAWANI_API_KEY,
      currency: 'SAR',
      region: 'SA'
    });

    this.registerGateway({
      name: 'MyFatoorah',
      apiKey: process.env.MYFATOORAH_API_KEY,
      enabled: !!process.env.MYFATOORAH_API_KEY,
      currency: 'KWD',
      region: 'KW'
    });

    this.registerGateway({
      name: 'HyperPay',
      apiKey: process.env.HYPERPAY_API_KEY,
      enabled: !!process.env.HYPERPAY_API_KEY,
      currency: 'SAR',
      region: 'SA'
    });

    this.registerGateway({
      name: 'PayTabs',
      merchantId: process.env.PAYTABS_MERCHANT_EMAIL,
      apiKey: process.env.PAYTABS_API_KEY,
      enabled: !!process.env.PAYTABS_API_KEY,
      currency: 'AED',
      region: 'AE'
    });

    this.registerGateway({
      name: 'Network International',
      apiKey: process.env.NGENIUS_API_KEY,
      enabled: !!process.env.NGENIUS_API_KEY,
      currency: 'AED',
      region: 'AE'
    });

    // Europe
    this.registerGateway({
      name: 'Iyzico',
      apiKey: process.env.IYZICO_API_KEY,
      enabled: !!process.env.IYZICO_API_KEY,
      currency: 'TRY',
      region: 'TR'
    });

    this.registerGateway({
      name: 'PayTR',
      merchantId: process.env.PAYTR_MERCHANT_ID,
      apiKey: process.env.PAYTR_API_KEY,
      enabled: !!process.env.PAYTR_API_KEY,
      currency: 'TRY',
      region: 'TR'
    });

    // Africa
    this.registerGateway({
      name: 'PayMe',
      apiKey: process.env.PAYME_API_KEY,
      enabled: !!process.env.PAYME_API_KEY,
      currency: 'UZS',
      region: 'UZ'
    });

    this.registerGateway({
      name: 'Kaspi Pay',
      apiKey: process.env.KASPI_API_KEY,
      enabled: !!process.env.KASPI_API_KEY,
      currency: 'KZT',
      region: 'KZ'
    });

    // Add remaining gateways (abbreviated for space)
    const otherGateways = [
      { name: 'Omise', region: 'TH', currency: 'THB' },
      { name: '2C2P', region: 'TH', currency: 'THB' },
      { name: 'ABA PAY', region: 'KH', currency: 'KHR' },
      { name: 'WingPay', region: 'KH', currency: 'KHR' },
      { name: 'OVO', region: 'ID', currency: 'IDR' },
      { name: 'BCEL OnePay', region: 'LA', currency: 'LAK' },
      { name: 'Touch n Go eWallet', region: 'MY', currency: 'MYR' },
      { name: 'KBZPay', region: 'MM', currency: 'MMK' },
      { name: 'ZaloPay', region: 'VN', currency: 'VND' },
      { name: 'OnePay Laos', region: 'LA', currency: 'LAK' },
      { name: 'iPay88', region: 'MY', currency: 'MYR' },
      { name: 'WavePay', region: 'MM', currency: 'MMK' },
      { name: 'NewebPay', region: 'TW', currency: 'TWD' },
      { name: 'Toss Payments', region: 'KR', currency: 'KRW' },
      { name: 'PayPay', region: 'JP', currency: 'JPY' },
      { name: 'Rakuten Pay', region: 'JP', currency: 'JPY' },
      { name: 'Kakao Pay', region: 'KR', currency: 'KRW' }
    ];

    otherGateways.forEach((gw) => {
      this.registerGateway({
        name: gw.name,
        enabled: false, // Enable when credentials added
        currency: gw.currency,
        region: gw.region
      });
    });
  }

  private registerGateway(config: PaymentGatewayConfig) {
    this.gateways.set(config.name.toLowerCase(), config);
  }

  // Main payment creation method
  async createPaymentSession(
    gatewayName: string,
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string,
    customerName: string
  ): Promise<PaymentSession> {
    const gateway = this.gateways.get(gatewayName.toLowerCase());

    if (!gateway || !gateway.enabled) {
      throw new BadRequestException(`Payment gateway ${gatewayName} not available`);
    }

    this.logger.log(`Creating payment session with ${gatewayName} for order ${orderId}`);

    switch (gatewayName.toLowerCase()) {
      case 'stripe':
        return this.createStripeSession(amount, currency, orderId, customerEmail);

      case 'sslcommerz':
        return this.createSSLCommerceSession(amount, currency, orderId, customerEmail, customerName);

      case 'bkash pg':
        return this.createBkashSession(amount, currency, orderId);

      case 'razorpay':
        return this.createRazorpaySession(amount, currency, orderId, customerEmail);

      case 'paymongo':
        return this.createPayMongoSession(amount, currency, orderId);

      case 'khalti':
        return this.createKhaltiSession(amount, currency, orderId, customerEmail);

      case 'xendit':
        return this.createXenditSession(amount, currency, orderId);

      case 'payhere':
        return this.createPayHereSession(amount, currency, orderId, customerEmail);

      case 'paytabs':
        return this.createPayTabsSession(amount, currency, orderId, customerEmail);

      case 'thawani pay':
        return this.createThawaniSession(amount, currency, orderId);

      case 'myfatoorah':
        return this.createMyFatoorahSession(amount, currency, orderId);

      case 'paypal':
        return this.createPayPalSession(amount, currency, orderId);

      case 'alipay':
        return this.createAlipaySession(amount, currency, orderId);

      case 'wechat pay':
        return this.createWechatSession(amount, currency, orderId);

      default:
        throw new BadRequestException(`Payment gateway ${gatewayName} not implemented yet`);
    }
  }

  // Get available gateways for region/currency
  async getAvailableGateways(region?: string, currency?: string): Promise<PaymentGatewayConfig[]> {
    const available = Array.from(this.gateways.values()).filter((gw) => {
      if (!gw.enabled) return false;
      if (region && gw.region && gw.region !== 'global' && gw.region !== region) return false;
      if (currency && gw.currency !== currency) return false;
      return true;
    });

    return available;
  }

  // ===== GATEWAY IMPLEMENTATIONS =====

  private async createStripeSession(
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string
  ): Promise<PaymentSession> {
    if (!this.stripe) {
      this.stripe = new Stripe(process.env.STRIPE_SECRET_KEY, {
        apiVersion: '2023-10-16'
      });
    }

    const session = await this.stripe.checkout.sessions.create({
      payment_method_types: ['card'],
      line_items: [
        {
          price_data: {
            currency: currency.toLowerCase(),
            product_data: { name: `Order ${orderId}` },
            unit_amount: Math.round(amount * 100)
          },
          quantity: 1
        }
      ],
      mode: 'payment',
      success_url: `${process.env.APP_URL}/payment/success?session_id={CHECKOUT_SESSION_ID}`,
      cancel_url: `${process.env.APP_URL}/payment/cancel`,
      customer_email: customerEmail,
      metadata: { orderId }
    });

    return {
      gatewayName: 'Stripe',
      sessionId: session.id,
      checkoutUrl: session.url,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createSSLCommerceSession(
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string,
    customerName: string
  ): Promise<PaymentSession> {
    const gateway = this.gateways.get('sslcommerz');

    return {
      gatewayName: 'SSLCommerz',
      sessionId: orderId,
      redirectUrl: 'https://sandbox.sslcommerz.com/gwprocess/v4/api.php',
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createBkashSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement bKash API integration
    // https://developer.bkash.com/

    return {
      gatewayName: 'bKash PG',
      sessionId: orderId,
      redirectUrl: 'https://checkout.bkash.com',
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createRazorpaySession(
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string
  ): Promise<PaymentSession> {
    // Implement Razorpay API integration
    // https://razorpay.com/docs/

    return {
      gatewayName: 'Razorpay',
      sessionId: orderId,
      checkoutUrl: `https://checkout.razorpay.com/?key_id=${process.env.RAZORPAY_KEY_ID}&order_id=${orderId}`,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createPayMongoSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement PayMongo API integration
    const response = await axios.post(
      'https://api.paymongo.com/v1/checkout_sessions',
      {
        data: {
          attributes: {
            amount: Math.round(amount * 100),
            currency: currency.toLowerCase(),
            description: `Order ${orderId}`
          }
        }
      },
      {
        headers: {
          Authorization: `Basic ${Buffer.from(
            process.env.PAYMONGO_SECRET_KEY + ':'
          ).toString('base64')}`
        }
      }
    );

    return {
      gatewayName: 'PayMongo',
      sessionId: response.data.data.id,
      checkoutUrl: response.data.data.attributes.checkout_url,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createKhaltiSession(
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string
  ): Promise<PaymentSession> {
    const response = await axios.post(
      'https://khalti.com/api/payment/initiate/',
      {
        public_key: process.env.KHALTI_PUBLIC_KEY,
        amount: Math.round(amount * 100),
        purchase_order_id: orderId,
        return_url: `${process.env.APP_URL}/payment/khalti/verify`
      }
    );

    return {
      gatewayName: 'Khalti',
      sessionId: response.data.payment_url.split('?')[1],
      redirectUrl: response.data.payment_url,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createXenditSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement Xendit API integration
    const response = await axios.post(
      'https://api.xendit.co/v2/invoices',
      {
        external_id: orderId,
        amount: Math.round(amount),
        currency: currency,
        success_redirect_url: `${process.env.APP_URL}/payment/success`,
        failure_redirect_url: `${process.env.APP_URL}/payment/cancel`
      },
      {
        auth: {
          username: process.env.XENDIT_API_KEY || '',
          password: ''
        }
      }
    );

    return {
      gatewayName: 'Xendit',
      sessionId: response.data.id,
      redirectUrl: response.data.invoice_url,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createPayHereSession(
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string
  ): Promise<PaymentSession> {
    // Implement PayHere API integration
    // https://www.payhere.lk/

    return {
      gatewayName: 'PayHere',
      sessionId: orderId,
      redirectUrl: 'https://sandbox.payhere.lk/pay/checkout',
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createPayTabsSession(
    amount: number,
    currency: string,
    orderId: string,
    customerEmail: string
  ): Promise<PaymentSession> {
    // Implement PayTabs API integration
    const response = await axios.post(
      'https://www.paytabs.com/api/create_pay_page',
      {
        merchant_email: process.env.PAYTABS_MERCHANT_EMAIL,
        secret_key: process.env.PAYTABS_API_KEY,
        cc_currency: currency,
        cc_amount: amount,
        cc_customer_email: customerEmail,
        cc_product_id: orderId,
        return_url: `${process.env.APP_URL}/payment/success`,
        cancel_url: `${process.env.APP_URL}/payment/cancel`
      }
    );

    return {
      gatewayName: 'PayTabs',
      sessionId: response.data.pay_page_id,
      redirectUrl: response.data.redirect_url,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createThawaniSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement Thawani API integration
    const response = await axios.post(
      'https://api.thawani.iq/v1/checkout/session',
      {
        client_reference_id: orderId,
        products: [
          {
            name: `Order ${orderId}`,
            unit_amount: Math.round(amount * 1000),
            quantity: 1
          }
        ],
        currency: currency.toLowerCase(),
        success_url: `${process.env.APP_URL}/payment/success`,
        cancel_url: `${process.env.APP_URL}/payment/cancel`
      },
      {
        headers: {
          Authorization: `Bearer ${process.env.THAWANI_API_KEY}`
        }
      }
    );

    return {
      gatewayName: 'Thawani Pay',
      sessionId: response.data.session_id,
      redirectUrl: response.data.redirect_url,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createMyFatoorahSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement MyFatoorah API integration
    const response = await axios.post(
      'https://apitest.myfatoorah.com/v2/SendPayment',
      {
        InvoiceAmount: amount,
        CurrencyIso: currency,
        OrderId: orderId,
        NotificationOption: 'Email'
      },
      {
        headers: {
          Authorization: `Bearer ${process.env.MYFATOORAH_API_KEY}`
        }
      }
    );

    return {
      gatewayName: 'MyFatoorah',
      sessionId: response.data.InvoiceId,
      redirectUrl: response.data.Data.PaymentURL,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createPayPalSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement PayPal API integration
    // https://developer.paypal.com/

    return {
      gatewayName: 'PayPal',
      sessionId: orderId,
      redirectUrl: `https://www.sandbox.paypal.com/checkoutnow`,
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createAlipaySession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement Alipay API integration
    // https://open.alipay.com/

    return {
      gatewayName: 'Alipay',
      sessionId: orderId,
      redirectUrl: 'https://openapi.alipay.com/gateway.do',
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  private async createWechatSession(
    amount: number,
    currency: string,
    orderId: string
  ): Promise<PaymentSession> {
    // Implement WeChat Pay API integration
    // https://pay.weixin.qq.com/

    return {
      gatewayName: 'WeChat Pay',
      sessionId: orderId,
      redirectUrl: 'https://api.mch.weixin.qq.com/pay/unifiedorder',
      amount,
      currency,
      orderId,
      status: 'pending'
    };
  }

  // Smart gateway selection based on region/currency
  async selectBestGateway(
    region: string,
    currency: string,
    amount: number
  ): Promise<PaymentGatewayConfig> {
    const available = await this.getAvailableGateways(region, currency);

    if (available.length === 0) {
      // Fallback to global gateways
      const globalGateways = await this.getAvailableGateways();
      if (globalGateways.length === 0) {
        throw new BadRequestException('No payment gateways available');
      }
      return globalGateways[0];
    }

    // Prefer region-specific gateway
    const regionSpecific = available.find((gw) => gw.region === region);
    return regionSpecific || available[0];
  }
}