import {
  Controller,
  Post,
  Get,
  Body,
  Param,
  Query,
  UseGuards,
  Req,
  BadRequestException
} from '@nestjs/common';
import { ApiTags, ApiBearerAuth } from '@nestjs/swagger';
import { PaymentService } from './payment.service';
import { PaymentGatewayService } from './payment-gateway.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PrismaService } from '../../database/prisma.service';

@ApiTags('Payments')
@Controller('payments')
export class PaymentController {
  constructor(
    private paymentService: PaymentService,
    private gatewayService: PaymentGatewayService,
    private prisma: PrismaService
  ) {}

  @Get('gateways')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async getAvailableGateways(
    @Req() req: any,
    @Query('region') region?: string,
    @Query('currency') currency?: string
  ) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: req.user.tenantId }
    });

    return {
      success: true,
      data: await this.gatewayService.getAvailableGateways(
        region || tenant?.country,
        currency || tenant?.currency
      )
    };
  }

  @Post('session')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async createPaymentSession(
    @Req() req: any,
    @Body() body: { orderId: string; gatewayName?: string }
  ) {
    const order = await this.prisma.order.findFirst({
      where: { id: body.orderId, tenantId: req.user.tenantId },
      include: { user: true }
    });

    if (!order) {
      throw new BadRequestException('Order not found');
    }

    const tenant = await this.prisma.tenant.findUnique({
      where: { id: req.user.tenantId }
    });

    let gateway = body.gatewayName
      ? { name: body.gatewayName }
      : await this.gatewayService.selectBestGateway(
          tenant?.country || 'US',
          tenant?.currency || 'USD',
          order.total
        );

    const session = await this.gatewayService.createPaymentSession(
      gateway.name,
      order.total,
      tenant?.currency || 'USD',
      order.id,
      order.user.email,
      order.user.name
    );

    return {
      success: true,
      data: session
    };
  }

  @Get('session/:sessionId')
  async getSessionStatus(@Param('sessionId') sessionId: string) {
    // Verify with payment gateway
    return {
      success: true,
      data: { status: 'completed' }
    };
  }

  @Post('verify/:gatewayName')
  async verifyPayment(
    @Param('gatewayName') gatewayName: string,
    @Body() body: any
  ) {
    // Verify with specific gateway
    // Call appropriate webhook handler based on gateway name
    return { success: true };
  }

  @Post('webhook/:gatewayName')
  async handleWebhook(
    @Param('gatewayName') gatewayName: string,
    @Body() body: any,
    @Req() req: any
  ) {
    // Route to appropriate webhook handler
    switch (gatewayName.toLowerCase()) {
      case 'stripe':
        return await this.paymentService.handleStripeWebhook(body);
      case 'sslcommerz':
        return await this.paymentService.handleSSLCommerceWebhook(body);
      case 'razorpay':
        return await this.paymentService.handleRazorpayWebhook(body);
      case 'paymongo':
        return await this.paymentService.handlePayMongoWebhook(body);
      default:
        return { success: false, message: 'Unknown gateway' };
    }
  }
}