import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class PaymentService {
  private logger = new Logger('PaymentService');

  constructor(private prisma: PrismaService) {}

  // দেশ-ভিত্তিক পেমেন্ট গেটওয়ে নির্বাচন
  async getAvailablePaymentMethods(country: string) {
    const paymentMethods: any = {
      BD: [
        { name: 'bKash', type: 'mobile_money' },
        { name: 'Nagad', type: 'mobile_money' },
        { name: 'SSLCOMMERZ', type: 'card' }
      ],
      US: [
        { name: 'Stripe', type: 'card' },
        { name: 'PayPal', type: 'wallet' }
      ],
      IN: [
        { name: 'Razorpay', type: 'card' },
        { name: 'PayPal', type: 'wallet' }
      ]
    };

    return paymentMethods[country] || paymentMethods['US'];
  }

  // bKash পেমেন্ট প্রসেস করুন
  async processBKashPayment(orderId: string, amount: number, phone: string) {
    try {
      const response = await axios.post(
        'https://api.bkash.com/payment/initiate',
        {
          amount,
          phone,
          reference: orderId,
          callback: `${process.env.APP_URL}/payment/bkash/callback`
        },
        { headers: { Authorization: `Bearer ${process.env.BKASH_API_KEY}` } }
      );

      return { success: true, paymentUrl: response.data.paymentUrl };
    } catch (err) {
      this.logger.error('bKash payment failed:', err);
      throw err;
    }
  }

  // Stripe পেমেন্ট প্রসেস করুন
  async processStripePayment(orderId: string, amount: number, token: string) {
    try {
      // Stripe API কল করুন
      const response = await axios.post(
        'https://api.stripe.com/v1/charges',
        {
          amount: amount * 100, // সেন্টে রূপান্তর
          currency: 'usd',
          source: token,
          description: `Order ${orderId}`
        },
        {
          auth: {
            username: process.env.STRIPE_SECRET_KEY || '',
            password: ''
          }
        }
      );

      if (response.data.status === 'succeeded') {
        // অর্ডার নিশ্চিত করুন
        await this.prisma.order.update({
          where: { id: orderId },
          data: { status: 'CONFIRMED', paymentId: response.data.id }
        });

        return { success: true, paymentId: response.data.id };
      }
    } catch (err) {
      this.logger.error('Stripe payment failed:', err);
      throw err;
    }
  }

  // পেমেন্ট যাচাই করুন
  async verifyPayment(orderId: string, paymentId: string) {
    const order = await this.prisma.order.findUnique({
      where: { id: orderId }
    });

    return {
      status: order?.status,
      paymentId,
      amount: order?.total
    };
  }
}