import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class SettlementService {
  private logger = new Logger('SettlementService');

  constructor(private prisma: PrismaService) {}

  // মাসিক সেটলমেন্ট জেনারেট করুন
  @Cron('0 0 1 * *') // মাসের প্রথম দিন
  async generateMonthlySettlement() {
    const wholesalers = await this.prisma.wholesaler.findMany({
      where: { status: 'active' }
    });

    for (const wholesaler of wholesalers) {
      await this.calculateSettlement(wholesaler.id);
    }
  }

  private async calculateSettlement(wholesalerId: string) {
    const lastMonth = new Date();
    lastMonth.setMonth(lastMonth.getMonth() - 1);

    // মাসিক বিক্রয় পান
    const orders = await this.prisma.order.findMany({
      where: {
        items: {
          some: {
            wholesalerId
          }
        },
        createdAt: {
          gte: new Date(lastMonth.getFullYear(), lastMonth.getMonth(), 1),
          lt: new Date(lastMonth.getFullYear(), lastMonth.getMonth() + 1, 1)
        }
      }
    });

    const totalEarnings = orders.reduce((sum, o) => sum + o.total, 0);
    const platformFee = totalEarnings * 0.1; // 10% প্ল্যাটফর্ম ফি
    const payableAmount = totalEarnings - platformFee;

    const period = `${lastMonth.getFullYear()}-${String(lastMonth.getMonth() + 1).padStart(2, '0')}`;

    const settlement = await this.prisma.paymentSettlement.create({
      data: {
        wholesalerTenantId: wholesalerId,
        period,
        totalEarnings,
        platformFee,
        payableAmount,
        status: 'pending'
      }
    });

    this.logger.log(`Settlement created: ${wholesalerId} - $${payableAmount}`);

    return settlement;
  }

  // সেটলমেন্ট মার্ক করুন পেইড হিসেবে
  async markSettlementAsPaid(settlementId: string, transactionId: string) {
    const settlement = await this.prisma.paymentSettlement.update({
      where: { id: settlementId },
      data: {
        status: 'paid',
        paidAt: new Date(),
        transactionId
      }
    });

    return { success: true, data: settlement };
  }
}