import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class PriceTierService {
  constructor(private prisma: PrismaService) {}

  // মূল্য স্তর যোগ করুন
  async createPriceTier(
    tenantId: string,
    input: {
      name: string;
      minQuantity: number;
      maxQuantity?: number;
      discountPercent: number;
    }
  ) {
    const tier = await this.prisma.priceTier.create({
      data: {
        tenantId,
        name: input.name,
        minQuantity: input.minQuantity,
        maxQuantity: input.maxQuantity,
        discountPercent: input.discountPercent,
        discountType: 'percentage'
      }
    });

    return { success: true, data: tier };
  }

  // কোয়ান্টিটি অনুযায়ী মূল্য পান
  async getPriceForQuantity(
    tenantId: string,
    basePrice: number,
    quantity: number
  ): Promise<number> {
    const tier = await this.prisma.priceTier.findFirst({
      where: {
        tenantId,
        minQuantity: { lte: quantity },
        OR: [
          { maxQuantity: null },
          { maxQuantity: { gte: quantity } }
        ]
      },
      orderBy: { discountPercent: 'desc' } // সর্বোচ্চ ছাড় পান
    });

    if (!tier) return basePrice;

    const discount = (basePrice * tier.discountPercent) / 100;
    return basePrice - discount;
  }
}