import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { InjectQueue } from '@nestjs/bull';
import { Queue } from 'bull';

@Injectable()
export class BulkEditorService {
  private logger = new Logger('BulkEditorService');

  constructor(
    private prisma: PrismaService,
    @InjectQueue('bulk-operations') private bulkQueue: Queue
  ) {}

  // বাল্ক এডিট শুরু করুন
  async bulkEdit(
    tenantId: string,
    productIds: string[],
    editType: string,
    changes: any,
    userId: string
  ) {
    // সিতিহাস সংরক্ষণ করুন
    const history = await this.prisma.bulkEditHistory.create({
      data: {
        tenantId,
        productIds,
        editType,
        changes,
        status: 'pending',
        createdBy: userId
      }
    });

    // বাল্ক কাজ কিউতে যোগ করুন
    await this.bulkQueue.add('bulk-edit', {
      historyId: history.id,
      productIds,
      editType,
      changes,
      tenantId
    });

    return { success: true, historyId: history.id };
  }

  // বাল্ক প্রাইস আপডেট
  async bulkUpdatePrice(tenantId: string, productIds: string[], priceChange: any) {
    if (priceChange.type === 'fixed') {
      // ফিক্সড প্রাইস
      await this.prisma.product.updateMany({
        where: { id: { in: productIds }, tenantId },
        data: { price: priceChange.value }
      });
    } else if (priceChange.type === 'percentage') {
      // পার্সেন্টেজ বৃদ্ধি/হ্রাস
      const products = await this.prisma.product.findMany({
        where: { id: { in: productIds }, tenantId }
      });

      for (const product of products) {
        const newPrice = product.price * (1 + priceChange.value / 100);
        await this.prisma.product.update({
          where: { id: product.id },
          data: { price: newPrice }
        });
      }
    }

    this.logger.log(`Bulk price update: ${productIds.length} products`);

    return { success: true, updatedCount: productIds.length };
  }

  // বাল্ক ক্যাটেগরি আপডেট
  async bulkUpdateCategory(tenantId: string, productIds: string[], categoryId: string) {
    await this.prisma.product.updateMany({
      where: { id: { in: productIds }, tenantId },
      data: { categoryId }
    });

    return { success: true, updatedCount: productIds.length };
  }

  // বাল্ক স্টক আপডেট
  async bulkUpdateStock(tenantId: string, productIds: string[], stockChange: any) {
    if (stockChange.type === 'set') {
      await this.prisma.product.updateMany({
        where: { id: { in: productIds }, tenantId },
        data: { stock: stockChange.value }
      });
    } else if (stockChange.type === 'add') {
      const products = await this.prisma.product.findMany({
        where: { id: { in: productIds }, tenantId }
      });

      for (const product of products) {
        await this.prisma.product.update({
          where: { id: product.id },
          data: { stock: product.stock + stockChange.value }
        });
      }
    }

    return { success: true, updatedCount: productIds.length };
  }

  // এডিট হিস্ট্রি পান
  async getEditHistory(tenantId: string, limit = 50) {
    const history = await this.prisma.bulkEditHistory.findMany({
      where: { tenantId },
      orderBy: { createdAt: 'desc' },
      take: limit
    });

    return { success: true, data: history };
  }
}