import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as csv from 'csv-parse/sync';
import axios from 'axios';

@Injectable()
export class BulkImportService {
  private logger = new Logger('BulkImportService');

  constructor(private prisma: PrismaService) {}

  // CSV থেকে পণ্য ইমপোর্ট করুন
  async importFromCSV(tenantId: string, csvContent: string) {
    const records = csv.parse(csvContent, {
      columns: true,
      skip_empty_lines: true
    });

    const products = [];

    for (const record of records) {
      const product = await this.prisma.product.create({
        data: {
          tenantId,
          sku: record.sku,
          title: record.title,
          description: record.description,
          price: parseFloat(record.price),
          stock: parseInt(record.stock) || 0,
          categoryId: record.categoryId
        }
      });

      products.push(product);
    }

    this.logger.log(`Imported ${products.length} products from CSV`);

    return { success: true, importedCount: products.length };
  }

  // হোলসেলার সাইট থেকে স্ক্র্যাপ করুন (Merrono, Dropshipping.com.bd)
  async importFromWholesaler(tenantId: string, wholesalerId: string) {
    try {
      // Merrono API থেকে পণ্য পান
      const response = await axios.get(
        `https://api.merrono.com/products?wholesaler_id=${wholesalerId}`,
        {
          headers: { Authorization: `Bearer ${process.env.MERRONO_API_KEY}` }
        }
      );

      const products = response.data.items || [];
      const imported = [];

      for (const item of products) {
        try {
          const product = await this.prisma.product.create({
            data: {
              tenantId,
              sku: item.sku,
              title: item.title,
              description: item.description,
              price: item.price,
              stock: item.stock || 0,
              images: item.images || []
            }
          });

          imported.push(product);
        } catch (err) {
          this.logger.warn(`Failed to import product ${item.sku}`);
        }
      }

      return { success: true, importedCount: imported.length };
    } catch (err) {
      this.logger.error('Wholesaler import failed:', err);
      throw err;
    }
  }
}