import { Injectable, Logger } from '@nestjs/common';
import * as sharp from 'sharp';
import * as AWS from 'aws-sdk';

@Injectable()
export class ProductImageService {
  private logger = new Logger('ProductImageService');
  private s3 = new AWS.S3();

  // ইমেজ আপলোড করুন (WebP কনভার্সন সহ)
  async uploadProductImage(tenantId: string, imageFile: Buffer): Promise<string[]> {
    try {
      const originalUrl = await this.uploadToS3(
        tenantId,
        `original-${Date.now()}.jpg`,
        imageFile,
        'image/jpeg'
      );

      // WebP তৈরি করুন
      const webpBuffer = await sharp(imageFile)
        .webp({ quality: 80 })
        .toBuffer();

      const webpUrl = await this.uploadToS3(
        tenantId,
        `webp-${Date.now()}.webp`,
        webpBuffer,
        'image/webp'
      );

      return [originalUrl, webpUrl];
    } catch (err) {
      this.logger.error('Image upload failed:', err);
      throw err;
    }
  }

  private async uploadToS3(
    tenantId: string,
    filename: string,
    buffer: Buffer,
    contentType: string
  ): Promise<string> {
    const key = `products/${tenantId}/${filename}`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: buffer,
        ContentType: contentType,
        CacheControl: 'max-age=31536000' // 1 বছর cache
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }
}