import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as csv from 'csv-parse/sync';
import * as XLSX from 'xlsx';
import axios from 'axios';
import * as AWS from 'aws-sdk';

@Injectable()
export class ProductImportService {
  private logger = new Logger('ProductImportService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // প্রোডাক্ট ইমপোর্ট শুরু করুন
  async startImport(
    tenantId: string,
    fileBuffer: Buffer,
    fileName: string,
    fileType: 'csv' | 'excel' | 'json'
  ) {
    try {
      // ইমপোর্ট জব তৈরি করুন
      const importJob = await this.prisma.productImportJob.create({
        data: {
          tenantId,
          fileName,
          fileUrl: '', // Will update after S3 upload
          fileType,
          totalRows: 0,
          status: 'pending'
        }
      });

      // ফাইল S3-এ আপলোড করুন
      const s3Key = `imports/${tenantId}/${importJob.id}-${fileName}`;
      await this.s3
        .putObject({
          Bucket: process.env.AWS_S3_BUCKET || '',
          Key: s3Key,
          Body: fileBuffer
        })
        .promise();

      // ডেটা পার্স করুন
      let records: any[] = [];

      if (fileType === 'csv') {
        records = csv.parse(fileBuffer.toString(), {
          columns: true,
          skip_empty_lines: true
        });
      } else if (fileType === 'excel') {
        const workbook = XLSX.read(fileBuffer);
        const sheet = workbook.Sheets[workbook.SheetNames[0]];
        records = XLSX.utils.sheet_to_json(sheet);
      } else if (fileType === 'json') {
        records = JSON.parse(fileBuffer.toString());
      }

      // ইমপোর্ট জব আপডেট করুন
      await this.prisma.productImportJob.update({
        where: { id: importJob.id },
        data: {
          fileUrl: `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${s3Key}`,
          totalRows: records.length,
          status: 'processing'
        }
      });

      // ব্যাকগ্রাউন্ডে প্রসেস করুন
      this.processImportAsync(importJob.id, tenantId, records);

      return {
        success: true,
        jobId: importJob.id,
        totalRows: records.length
      };
    } catch (err) {
      this.logger.error('Import failed:', err);
      throw err;
    }
  }

  private async processImportAsync(jobId: string, tenantId: string, records: any[]) {
    let successCount = 0;
    let failureCount = 0;
    const errors: string[] = [];

    for (const record of records) {
      try {
        // প্রোডাক্ট ডেটা ভ্যালিডেট করুন
        if (!record.sku || !record.title || !record.price) {
          failureCount++;
          errors.push(`Row: Missing required fields (SKU, Title, Price)`);
          continue;
        }

        // প্রোডাক্ট তৈরি করুন
        await this.prisma.product.upsert({
          where: { tenantId_sku: { tenantId, sku: record.sku } },
          create: {
            tenantId,
            sku: record.sku,
            title: record.title,
            description: record.description || '',
            price: parseFloat(record.price),
            stock: parseInt(record.stock) || 0,
            categoryId: record.categoryId,
            images: record.images ? record.images.split('|') : []
          },
          update: {
            title: record.title,
            description: record.description || '',
            price: parseFloat(record.price),
            stock: parseInt(record.stock) || 0,
            categoryId: record.categoryId
          }
        });

        successCount++;
      } catch (err) {
        failureCount++;
        errors.push(`Row ${record.sku}: ${(err as Error).message}`);
      }
    }

    // জব সম্পন্ন করুন
    await this.prisma.productImportJob.update({
      where: { id: jobId },
      data: {
        status: 'completed',
        successCount,
        failureCount,
        errors,
        completedAt: new Date()
      }
    });

    this.logger.log(`Import completed: ${successCount} success, ${failureCount} failed`);
  }

  // ইমপোর্ট স্ট্যাটাস পান
  async getImportStatus(jobId: string) {
    const job = await this.prisma.productImportJob.findUnique({
      where: { id: jobId }
    });

    return {
      success: true,
      data: job
    };
  }
}