import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class ProductService {
  private logger = new Logger('ProductService');

  constructor(private prisma: PrismaService) {}

  // পণ্য তৈরি করুন
  async createProduct(tenantId: string, input: {
    sku: string;
    title: string;
    description: string;
    price: number;
    stock: number;
    categoryId?: string;
    images?: string[];
  }) {
    const product = await this.prisma.product.create({
      data: {
        tenantId,
        sku: input.sku,
        title: input.title,
        description: input.description,
        price: input.price,
        stock: input.stock,
        categoryId: input.categoryId,
        images: input.images || []
      }
    });

    // SEO এন্ট্রি তৈরি করুন
    await this.prisma.productSeo.create({
      data: {
        productId: product.id,
        slug: this.generateSlug(input.title)
      }
    });

    return { success: true, data: product };
  }

  // ভেরিয়েশন যোগ করুন
  async addVariant(productId: string, input: {
    sku: string;
    name: string;
    attributes: Record<string, string>;
    price?: number;
    stock: number;
    images?: string[];
  }) {
    const variant = await this.prisma.productVariant.create({
      data: {
        productId,
        sku: input.sku,
        name: input.name,
        attributes: input.attributes,
        price: input.price,
        stock: input.stock,
        images: input.images || []
      }
    });

    return { success: true, data: variant };
  }

  // ক্যাটেগরি তৈরি করুন
  async createCategory(tenantId: string, name: string, parentId?: string) {
    const category = await this.prisma.productCategory.create({
      data: {
        tenantId,
        name,
        slug: this.generateSlug(name),
        parentId
      }
    });

    return { success: true, data: category };
  }

  // কালেকশন তৈরি করুন
  async createCollection(tenantId: string, input: {
    name: string;
    description?: string;
    productIds: string[];
  }) {
    const collection = await this.prisma.productCollection.create({
      data: {
        tenantId,
        name: input.name,
        slug: this.generateSlug(input.name),
        description: input.description,
        products: {
          connect: input.productIds.map((id) => ({ id }))
        }
      }
    });

    return { success: true, data: collection };
  }

  private generateSlug(text: string): string {
    return text
      .toLowerCase()
      .replace(/[^\w\s-]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-');
  }
}